/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.core.center.set;

import java.io.Serial;
import java.io.Serializable;
import java.util.*;
import java.util.function.Function;

import org.miaixz.bus.core.center.map.MapBuilder;
import org.miaixz.bus.core.xyz.ObjectKit;

/**
 * A {@link Set} implementation that ensures uniqueness of elements based on a custom unique key generated by a
 * {@link Function}. Unlike a standard {@link HashSet} which relies on {@link Object#equals(Object)} and
 * {@link Object#hashCode()}, this set allows defining a custom uniqueness rule externally, which is useful when objects
 * cannot be modified to override {@code equals} and {@code hashCode}.
 * <p>
 * Elements are stored in an internal {@link Map}, where the keys of the map are the unique keys generated by
 * {@link #uniqueGenerator}, and the values are the elements themselves.
 * 
 *
 * <pre>
 * {@code
 * 
 * Set<UniqueTestBean> set = new UniqueKeySet<>(UniqueTestBean::getId);
 * }
 * </pre>
 *
 * @param <K> The type of the unique key generated for each element.
 * @param <V> The type of the elements (values) in the set.
 * @author Kimi Liu
 * @since Java 17+
 */
public class UniqueKeySet<K, V> extends AbstractSet<V> implements Serializable {

    @Serial
    private static final long serialVersionUID = 2852280266036L;

    /**
     * The function used to generate a unique key for each element.
     */
    private final Function<V, K> uniqueGenerator;
    /**
     * The internal map that stores elements, using the generated unique keys.
     */
    private Map<K, V> map;

    /**
     * Constructs a new {@code UniqueKeySet} with the specified unique key generator. By default, it uses a
     * {@link HashMap} and does not maintain insertion order.
     *
     * @param uniqueGenerator The function to generate a unique key for each element. Must not be {@code null}.
     */
    public UniqueKeySet(final Function<V, K> uniqueGenerator) {
        this(false, uniqueGenerator);
    }

    /**
     * Constructs a new {@code UniqueKeySet} with the specified unique key generator and initial collection. By default,
     * it uses a {@link HashMap} and does not maintain insertion order.
     *
     * @param uniqueGenerator The function to generate a unique key for each element. Must not be {@code null}.
     * @param c               The collection whose elements are to be added to this set.
     */
    public UniqueKeySet(final Function<V, K> uniqueGenerator, final Collection<? extends V> c) {
        this(false, uniqueGenerator, c);
    }

    /**
     * Constructs a new {@code UniqueKeySet} with the specified unique key generator and order preference.
     *
     * @param isLinked        {@code true} to use a {@link LinkedHashMap} to maintain insertion order, {@code false} to
     *                        use a {@link HashMap}.
     * @param uniqueGenerator The function to generate a unique key for each element. Must not be {@code null}.
     */
    public UniqueKeySet(final boolean isLinked, final Function<V, K> uniqueGenerator) {
        this(MapBuilder.of(isLinked), uniqueGenerator);
    }

    /**
     * Constructs a new {@code UniqueKeySet} with the specified unique key generator, order preference, and initial
     * collection.
     *
     * @param isLinked        {@code true} to use a {@link LinkedHashMap} to maintain insertion order, {@code false} to
     *                        use a {@link HashMap}.
     * @param uniqueGenerator The function to generate a unique key for each element. Must not be {@code null}.
     * @param c               The collection whose elements are to be added to this set.
     */
    public UniqueKeySet(final boolean isLinked, final Function<V, K> uniqueGenerator, final Collection<? extends V> c) {
        this(isLinked, uniqueGenerator);
        addAll(c);
    }

    /**
     * Constructs a new {@code UniqueKeySet} with the specified initial capacity, load factor, and unique key generator.
     *
     * @param initialCapacity The initial capacity of the underlying map.
     * @param loadFactor      The load factor for the underlying map.
     * @param uniqueGenerator The function to generate a unique key for each element. Must not be {@code null}.
     */
    public UniqueKeySet(final int initialCapacity, final float loadFactor, final Function<V, K> uniqueGenerator) {
        this(MapBuilder.of(new HashMap<>(initialCapacity, loadFactor)), uniqueGenerator);
    }

    /**
     * Constructs a new {@code UniqueKeySet} with a custom {@link MapBuilder} and unique key generator.
     *
     * @param builder         The {@link MapBuilder} to create the underlying map. Must not be {@code null}.
     * @param uniqueGenerator The function to generate a unique key for each element. Must not be {@code null}.
     */
    public UniqueKeySet(final MapBuilder<K, V> builder, final Function<V, K> uniqueGenerator) {
        this.map = builder.build();
        this.uniqueGenerator = uniqueGenerator;
    }

    /**
     * Returns an iterator over the elements in this set.
     *
     * @return An {@link Iterator} over the elements in this set.
     */
    @Override
    public Iterator<V> iterator() {
        return map.values().iterator();
    }

    /**
     * Returns the number of elements in this set (its cardinality).
     *
     * @return The number of elements in this set.
     */
    @Override
    public int size() {
        return map.size();
    }

    /**
     * Returns {@code true} if this set contains no elements.
     *
     * @return {@code true} if this set contains no elements.
     */
    @Override
    public boolean isEmpty() {
        return map.isEmpty();
    }

    /**
     * Returns {@code true} if this set contains the specified element. The presence is determined by the unique key
     * generated for the element.
     *
     * @param o The element whose presence in this set is to be tested.
     * @return {@code true} if this set contains the specified element.
     */
    @Override
    public boolean contains(final Object o) {
        return map.containsKey(this.uniqueGenerator.apply((V) o));
    }

    /**
     * Adds the specified element to this set if it is not already present. The uniqueness is determined by the unique
     * key generated for the element.
     *
     * @param v The element to be added to this set.
     * @return {@code true} if this set did not already contain the specified element.
     */
    @Override
    public boolean add(final V v) {
        return null == map.put(this.uniqueGenerator.apply(v), v);
    }

    /**
     * Adds the specified element to this set if it is not already present. If the value already exists (based on its
     * unique key), it is ignored.
     *
     * @param v The element to be added to this set.
     * @return {@code true} if the element was successfully added (i.e., it was not already present).
     */
    public boolean addIfAbsent(final V v) {
        return null == map.putIfAbsent(this.uniqueGenerator.apply(v), v);
    }

    /**
     * Adds all of the elements in the specified collection to this set if they are not already present. If an element
     * already exists (based on its unique key), it is ignored.
     *
     * @param c The collection containing elements to be added to this set.
     * @return {@code true} if this set changed as a result of the call (i.e., one or more elements were added).
     */
    public boolean addAllIfAbsent(final Collection<? extends V> c) {
        boolean modified = false;
        for (final V v : c)
            if (addIfAbsent(v)) {
                modified = true;
            }
        return modified;
    }

    /**
     * Removes the specified element from this set if it is present. The element to remove is identified by its unique
     * key.
     *
     * @param o The element to be removed from this set, if present.
     * @return {@code true} if the set contained the specified element.
     */
    @Override
    public boolean remove(final Object o) {
        return null != map.remove(this.uniqueGenerator.apply((V) o));
    }

    /**
     * Removes all of the elements from this set. The set will be empty after this call returns.
     */
    @Override
    public void clear() {
        map.clear();
    }

    /**
     * Creates and returns a shallow copy of this {@code UniqueKeySet} instance. The underlying map is cloned, but the
     * elements themselves are not deep-copied.
     *
     * @return A shallow copy of this instance.
     * @throws InternalError if the map's class is not cloneable (which it is, by implementing {@link Cloneable}).
     */
    @Override
    public UniqueKeySet<K, V> clone() {
        try {
            final UniqueKeySet<K, V> newSet = (UniqueKeySet<K, V>) super.clone();
            newSet.map = ObjectKit.clone(this.map);
            return newSet;
        } catch (final CloneNotSupportedException e) {
            // This should not happen since we implement Cloneable.
            throw new InternalError(e);
        }
    }

}
