/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.core.center.stream;

import java.util.*;
import java.util.function.*;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.miaixz.bus.core.lang.Optional;
import org.miaixz.bus.core.lang.mutable.MutableInt;
import org.miaixz.bus.core.lang.mutable.MutableObject;
import org.miaixz.bus.core.xyz.ArrayKit;
import org.miaixz.bus.core.xyz.CollectorKit;

/**
 * An extension of {@link WrappedStream} that provides additional terminal operations for implementing classes. The
 * methods provided by this interface do not return a {@link Stream}.
 *
 * @param <T> the type of the elements in the stream
 * @param <S> the type of the {@link TerminableWrappedStream} implementation itself
 * @author Kimi Liu
 * @since Java 17+
 */
public interface TerminableWrappedStream<T, S extends TerminableWrappedStream<T, S>> extends WrappedStream<T, S> {

    /**
     * Collects the elements of this stream into a new {@link ArrayList}.
     *
     * @return a new {@link ArrayList} containing all elements of this stream
     * @see #toColl(Supplier)
     */
    default List<T> toList() {
        return this.toColl(ArrayList::new);
    }

    /**
     * Collects the elements of this stream into a new unmodifiable {@link List}.
     *
     * @return an unmodifiable {@link List} containing all elements of this stream
     * @see #toColl(Supplier)
     */
    default List<T> toUnmodifiableList() {
        return Collections.unmodifiableList(this.toList());
    }

    /**
     * Collects the elements of this stream into a new {@link HashSet}.
     *
     * @return a new {@link HashSet} containing all elements of this stream
     * @see #toColl(Supplier)
     */
    default Set<T> toSet() {
        return this.toColl(HashSet::new);
    }

    /**
     * Collects the elements of this stream into a new unmodifiable {@link Set}.
     *
     * @return an unmodifiable {@link Set} containing all elements of this stream
     * @see #toColl(Supplier)
     */
    default Set<T> toUnmodifiableSet() {
        return Collections.unmodifiableSet(this.toSet());
    }

    /**
     * Collects the elements of this stream into a new {@link Collection} supplied by the {@code collectionFactory}.
     *
     * @param collectionFactory a {@link Supplier} that returns a new, empty {@link Collection} of the appropriate type
     * @param <C>               the type of the {@link Collection}
     * @return a new {@link Collection} containing all elements of this stream
     */
    default <C extends Collection<T>> C toColl(final Supplier<C> collectionFactory) {
        Objects.requireNonNull(collectionFactory);
        return unwrap().collect(Collectors.toCollection(collectionFactory));
    }

    /**
     * Collects the elements of this stream into a {@link Map} where keys are generated by the {@code keyMapper} and
     * values are the elements themselves.
     *
     * @param keyMapper a function to extract the key from each element
     * @param <K>       the type of the keys in the map
     * @return a {@link Map} containing the collected elements
     * @see #toMap(Function, Function, BinaryOperator, Supplier)
     */
    default <K> Map<K, T> toMap(final Function<? super T, ? extends K> keyMapper) {
        return this.toMap(keyMapper, Function.identity());
    }

    /**
     * Collects the elements of this stream into a {@link Map} where keys are generated by the {@code keyMapper} and
     * values are generated by the {@code valueMapper}.
     *
     * @param keyMapper   a function to extract the key from each element
     * @param valueMapper a function to extract the value from each element
     * @param <K>         the type of the keys in the map
     * @param <U>         the type of the values in the map
     * @return a {@link Map} containing the collected elements
     * @see #toMap(Function, Function, BinaryOperator, Supplier)
     */
    default <K, U> Map<K, U> toMap(final Function<? super T, ? extends K> keyMapper,
            final Function<? super T, ? extends U> valueMapper) {
        return this.toMap(keyMapper, valueMapper, (l, r) -> r);
    }

    /**
     * Collects the elements of this stream into a new unmodifiable {@link Map} where keys are generated by the
     * {@code keyMapper} and values are generated by the {@code valueMapper}.
     *
     * @param keyMapper   a function to extract the key from each element
     * @param valueMapper a function to extract the value from each element
     * @param <K>         the type of the keys in the map
     * @param <U>         the type of the values in the map
     * @return an unmodifiable {@link Map} containing the collected elements
     * @see #toMap(Function, Function, BinaryOperator, Supplier)
     */
    default <K, U> Map<K, U> toUnmodifiableMap(final Function<? super T, ? extends K> keyMapper,
            final Function<? super T, ? extends U> valueMapper) {
        return Collections.unmodifiableMap(this.toMap(keyMapper, valueMapper));
    }

    /**
     * Collects the elements of this stream into a {@link Map} where keys are generated by the {@code keyMapper}, values
     * are generated by the {@code valueMapper}, and duplicate keys are resolved using the {@code mergeFunction}.
     *
     * @param keyMapper     a function to extract the key from each element
     * @param valueMapper   a function to extract the value from each element
     * @param mergeFunction a {@link BinaryOperator} to resolve collisions between values associated with the same key
     * @param <K>           the type of the keys in the map
     * @param <U>           the type of the values in the map
     * @return a {@link Map} containing the collected elements
     * @see #toMap(Function, Function, BinaryOperator, Supplier)
     */
    default <K, U> Map<K, U> toMap(final Function<? super T, ? extends K> keyMapper,
            final Function<? super T, ? extends U> valueMapper, final BinaryOperator<U> mergeFunction) {
        return this.toMap(keyMapper, valueMapper, mergeFunction, HashMap::new);
    }

    /**
     * Collects the elements of this stream into a new unmodifiable {@link Map} where keys are generated by the
     * {@code keyMapper}, values are generated by the {@code valueMapper}, and duplicate keys are resolved using the
     * {@code mergeFunction}.
     *
     * @param keyMapper     a function to extract the key from each element
     * @param valueMapper   a function to extract the value from each element
     * @param mergeFunction a {@link BinaryOperator} to resolve collisions between values associated with the same key
     * @param <K>           the type of the keys in the map
     * @param <U>           the type of the values in the map
     * @return an unmodifiable {@link Map} containing the collected elements
     * @see #toMap(Function, Function, BinaryOperator, Supplier)
     */
    default <K, U> Map<K, U> toUnmodifiableMap(final Function<? super T, ? extends K> keyMapper,
            final Function<? super T, ? extends U> valueMapper, final BinaryOperator<U> mergeFunction) {
        return Collections.unmodifiableMap(this.toMap(keyMapper, valueMapper, mergeFunction, HashMap::new));
    }

    /**
     * Collects the elements of this stream into a {@link Map} supplied by the {@code mapSupplier}, where keys are
     * generated by the {@code keyMapper}, values are generated by the {@code valueMapper}, and duplicate keys are
     * resolved using the {@code mergeFunction}.
     *
     * @param keyMapper     a function to extract the key from each element
     * @param valueMapper   a function to extract the value from each element
     * @param mergeFunction a {@link BinaryOperator} to resolve collisions between values associated with the same key
     * @param mapSupplier   a {@link Supplier} that returns a new, empty {@link Map} of the appropriate type
     * @param <K>           the type of the keys in the map
     * @param <U>           the type of the values in the map
     * @param <M>           the type of the {@link Map}
     * @return a {@link Map} containing the collected elements
     */
    default <K, U, M extends Map<K, U>> M toMap(final Function<? super T, ? extends K> keyMapper,
            final Function<? super T, ? extends U> valueMapper, final BinaryOperator<U> mergeFunction,
            final Supplier<M> mapSupplier) {
        Objects.requireNonNull(keyMapper);
        Objects.requireNonNull(valueMapper);
        Objects.requireNonNull(mergeFunction);
        Objects.requireNonNull(mapSupplier);
        return unwrap().collect(CollectorKit.toMap(keyMapper, valueMapper, mergeFunction, mapSupplier));
    }

    /**
     * Collects the elements of this stream into a {@link Map} where keys are the indices of the elements and values are
     * the elements themselves.
     *
     * @return a {@link Map} where keys are indices and values are elements
     */
    default Map<Integer, T> toIdxMap() {
        return toIdxMap(Function.identity());
    }

    /**
     * Collects the elements of this stream into a {@link Map} where keys are the indices of the elements and values are
     * generated by the {@code valueMapper}.
     *
     * @param valueMapper a function to extract the value from each element
     * @param <U>         the type of the values in the map
     * @return a {@link Map} where keys are indices and values are mapped elements
     */
    default <U> Map<Integer, U> toIdxMap(final Function<? super T, ? extends U> valueMapper) {
        final MutableInt index = new MutableInt(NOT_FOUND_ELEMENT_INDEX);
        return EasyStream.of(sequential().toList()).toMap(e -> index.incrementAndGet(), valueMapper, (l, r) -> r);
    }

    /**
     * Zips this stream with another {@link Iterable} to create a {@link Map}. The elements of this stream become the
     * keys, and the elements of the {@code other} iterable become the values. If the {@code other} iterable has fewer
     * elements than this stream, {@code null} will be used for missing values. If keys are duplicated, the last
     * associated value will be retained.
     *
     * @param other the {@link Iterable} to zip with this stream
     * @param <R>   the type of the elements in the {@code other} iterable (values in the resulting map)
     * @return a {@link Map} where keys are elements from this stream and values are elements from the {@code other}
     *         iterable
     */
    default <R> Map<T, R> toZip(final Iterable<R> other) {
        Objects.requireNonNull(other);
        // Value object iterator
        final Iterator<R> iterator = Optional.ofNullable(other).map(Iterable::iterator)
                .orElseGet(Collections::emptyIterator);
        if (this.isParallel()) {
            final List<T> keyList = toList();
            final Map<T, R> map = new HashMap<>(keyList.size());
            for (final T key : keyList) {
                map.put(key, iterator.hasNext() ? iterator.next() : null);
            }
            return map;
        } else {
            return this.toMap(Function.identity(), e -> iterator.hasNext() ? iterator.next() : null);
        }
    }

    /**
     * Finds the first element in this stream that matches the given predicate.
     *
     * @param predicate a {@link Predicate} to apply to elements of this stream
     * @return an {@link java.util.Optional} describing the first matching element, or an empty
     *         {@link java.util.Optional} if no element matches
     */
    default java.util.Optional<T> findFirst(final Predicate<? super T> predicate) {
        Objects.requireNonNull(predicate);
        return unwrap().filter(predicate).findFirst();
    }

    /**
     * Finds the index of the first element in this stream that matches the given predicate. For parallel streams, the
     * index will always be -1.
     *
     * @param predicate a {@link Predicate} to apply to elements of this stream
     * @return the index of the first matching element, or -1 if no element matches
     */
    default int findFirstIdx(final Predicate<? super T> predicate) {
        Objects.requireNonNull(predicate);
        final MutableInt index = new MutableInt(NOT_FOUND_ELEMENT_INDEX);
        unwrap().filter(e -> {
            index.increment();
            return predicate.test(e);
        }).findFirst(); // This is only for counting, not for value
        return index.get();
    }

    /**
     * Finds the last element in this stream.
     *
     * @return an {@link java.util.Optional} describing the last element, or an empty {@link java.util.Optional} if the
     *         stream is empty
     */
    default java.util.Optional<T> findLast() {
        final MutableObject<T> last = new MutableObject<>(null);
        spliterator().forEachRemaining(last::set);
        return java.util.Optional.ofNullable(last.get());
    }

    /**
     * Finds the last element in this stream that matches the given predicate.
     *
     * @param predicate a {@link Predicate} to apply to elements of this stream
     * @return an {@link java.util.Optional} describing the last matching element, or an empty
     *         {@link java.util.Optional} if no element matches
     */
    default java.util.Optional<T> findLast(final Predicate<? super T> predicate) {
        Objects.requireNonNull(predicate);
        final MutableObject<T> last = new MutableObject<>(null);
        spliterator().forEachRemaining(e -> {
            if (predicate.test(e)) {
                last.set(e);
            }
        });
        return java.util.Optional.ofNullable(last.get());
    }

    /**
     * Finds the index of the last element in this stream that matches the given predicate. For parallel streams, the
     * index will always be -1.
     *
     * @param predicate a {@link Predicate} to apply to elements of this stream
     * @return the index of the last matching element, or -1 if no element matches
     */
    default int findLastIdx(final Predicate<? super T> predicate) {
        Objects.requireNonNull(predicate);
        final MutableInt idxRef = new MutableInt(NOT_FOUND_ELEMENT_INDEX);
        forEachIdx((e, i) -> {
            if (predicate.test(e)) {
                idxRef.set(i);
            }
        });
        return idxRef.get();
    }

    /**
     * Retrieves the element at the specified index in the stream. If the index is negative, it counts from the end of
     * the stream.
     *
     * @param idx the index of the element to retrieve
     * @return an {@link java.util.Optional} describing the element at the specified index, or an empty
     *         {@link java.util.Optional} if the index is out of bounds
     */
    default java.util.Optional<T> at(final Integer idx) {
        return Optional.ofNullable(idx).map(i -> (T) ArrayKit.get(toArray(), i)).toOptional();
    }

    /**
     * Checks if this stream is empty.
     *
     * @return {@code true} if the stream is empty, {@code false} otherwise
     */
    default boolean isEmpty() {
        return !findAny().isPresent();
    }

    /**
     * Checks if this stream is not empty.
     *
     * @return {@code true} if the stream is not empty, {@code false} otherwise
     */
    default boolean isNotEmpty() {
        return !isEmpty();
    }

    /**
     * Joins the elements of this stream into a {@link String} without any delimiter, prefix, or suffix.
     *
     * @return a {@link String} consisting of the joined elements
     * @see #join(CharSequence, CharSequence, CharSequence)
     */
    default String join() {
        return this.join("");
    }

    /**
     * Joins the elements of this stream into a {@link String} using the specified delimiter.
     *
     * @param delimiter the delimiter to be used between each element
     * @return a {@link String} consisting of the joined elements
     * @see #join(CharSequence, CharSequence, CharSequence)
     */
    default String join(final CharSequence delimiter) {
        return this.join(delimiter, "", "");
    }

    /**
     * Joins the elements of this stream into a {@link String} using the specified delimiter, prefix, and suffix.
     *
     * @param delimiter the delimiter to be used between each element
     * @param prefix    the prefix to be used before the first element
     * @param suffix    the suffix to be used after the last element
     * @return a {@link String} consisting of the joined elements
     */
    default String join(final CharSequence delimiter, final CharSequence prefix, final CharSequence suffix) {
        return unwrap().map(String::valueOf)
                .collect(CollectorKit.joining(delimiter, prefix, suffix, Function.identity()));
    }

    /**
     * Groups the elements of this stream according to the provided classifier. Elements for which the classifier
     * returns {@code null} will not cause an exception.
     *
     * @param classifier a function to classify elements into groups
     * @param <K>        the type of the keys (group identifiers) in the resulting map
     * @return a {@link Map} where keys are the group identifiers and values are {@link List}s of elements belonging to
     *         that group
     * @see #group(Function, Supplier, Collector)
     */
    default <K> Map<K, List<T>> group(final Function<? super T, ? extends K> classifier) {
        return this.group(classifier, Collectors.toList());
    }

    /**
     * Groups the elements of this stream according to the provided classifier and collects them using the downstream
     * collector. Elements for which the classifier returns {@code null} will not cause an exception.
     *
     * @param classifier a function to classify elements into groups
     * @param downstream a {@link Collector} to accumulate the elements in each group
     * @param <K>        the type of the keys (group identifiers) in the resulting map
     * @param <D>        the result type of the downstream collector
     * @param <A>        the intermediate accumulation type of the downstream collector
     * @return a {@link Map} where keys are the group identifiers and values are the results of the downstream collector
     * @see #group(Function, Supplier, Collector)
     */
    default <K, A, D> Map<K, D> group(final Function<? super T, ? extends K> classifier,
            final Collector<? super T, A, D> downstream) {
        return this.group(classifier, HashMap::new, downstream);
    }

    /**
     * Groups the elements of this stream according to the provided classifier, collects them using the downstream
     * collector, and stores them in a {@link Map} supplied by the {@code mapFactory}. Elements for which the classifier
     * returns {@code null} will not cause an exception.
     *
     * @param classifier a function to classify elements into groups
     * @param mapFactory a {@link Supplier} that returns a new, empty {@link Map} of the appropriate type for the result
     * @param downstream a {@link Collector} to accumulate the elements in each group
     * @param <K>        the type of the keys (group identifiers) in the resulting map
     * @param <D>        the result type of the downstream collector
     * @param <A>        the intermediate accumulation type of the downstream collector
     * @param <M>        the type of the resulting {@link Map}
     * @return a {@link Map} where keys are the group identifiers and values are the results of the downstream collector
     * @see CollectorKit#groupingBy(Function, Supplier, Collector)
     */
    default <K, D, A, M extends Map<K, D>> M group(final Function<? super T, ? extends K> classifier,
            final Supplier<M> mapFactory, final Collector<? super T, A, D> downstream) {
        Objects.requireNonNull(classifier);
        Objects.requireNonNull(mapFactory);
        Objects.requireNonNull(downstream);
        return unwrap().collect(CollectorKit.groupingBy(classifier, mapFactory, downstream));
    }

    /**
     * Partitions the elements of this stream into two groups based on the given predicate. Elements that satisfy the
     * predicate are in one group (mapped to {@code true}), and those that don't are in the other (mapped to
     * {@code false}).
     *
     * @param predicate a {@link Predicate} to use for partitioning elements
     * @return a {@link Map} where keys are {@code Boolean} and values are {@link List}s of elements in each partition
     * @see #partition(Predicate, Collector)
     */
    default Map<Boolean, List<T>> partition(final Predicate<T> predicate) {
        return this.partition(predicate, ArrayList::new);
    }

    /**
     * Partitions the elements of this stream into two groups based on the given predicate, collecting elements into a
     * {@link Collection} supplied by the {@code collFactory}.
     *
     * @param <C>         the type of the {@link Collection} to hold the partitioned elements
     * @param predicate   a {@link Predicate} to use for partitioning elements
     * @param collFactory a {@link Supplier} that returns a new, empty {@link Collection} of the appropriate type for
     *                    each partition
     * @return a {@link Map} where keys are {@code Boolean} and values are {@link Collection}s of elements in each
     *         partition
     * @see #partition(Predicate, Collector)
     */
    default <C extends Collection<T>> Map<Boolean, C> partition(final Predicate<T> predicate,
            final Supplier<C> collFactory) {
        return this.partition(predicate, Collectors.toCollection(collFactory));
    }

    /**
     * Partitions the elements of this stream into two groups based on the given predicate, and collects them using the
     * downstream collector.
     *
     * @param predicate  a {@link Predicate} to use for partitioning elements
     * @param downstream a {@link Collector} to accumulate the elements in each partition
     * @param <R>        the result type of the downstream collector
     * @return a {@link Map} where keys are {@code Boolean} and values are the results of the downstream collector for
     *         each partition
     */
    default <R> Map<Boolean, R> partition(final Predicate<T> predicate, final Collector<T, ?, R> downstream) {
        Objects.requireNonNull(predicate);
        Objects.requireNonNull(downstream);
        return unwrap().collect(Collectors.partitioningBy(predicate, downstream));
    }

    /**
     * Performs an action for each element of this stream, providing the element's index. For parallel streams, the
     * index will always be -1. This is a terminal operation.
     *
     * @param action a {@link BiConsumer} to apply to each element and its index
     */
    default void forEachIdx(final BiConsumer<? super T, Integer> action) {
        Objects.requireNonNull(action);
        final MutableInt index = new MutableInt(NOT_FOUND_ELEMENT_INDEX);
        unwrap().forEach(e -> action.accept(e, index.incrementAndGet()));
    }

    /**
     * Performs an action for each element of this stream, in encounter order, providing the element's index. For
     * parallel streams, the index will always be -1. This is a terminal operation.
     *
     * @param action a {@link BiConsumer} to apply to each element and its index
     */
    default void forEachOrderedIdx(final BiConsumer<? super T, Integer> action) {
        Objects.requireNonNull(action);
        final MutableInt index = new MutableInt(NOT_FOUND_ELEMENT_INDEX);
        unwrap().forEachOrdered(e -> action.accept(e, index.incrementAndGet()));
    }

}
