/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.core.io.watch;

import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;

/**
 * Enumeration representing the types of watch events that can be monitored by a {@link java.nio.file.WatchService}.
 * These correspond to the standard watch event kinds defined in {@link java.nio.file.StandardWatchEventKinds}.
 *
 * <pre>
 *      1. OVERFLOW - {@link StandardWatchEventKinds#OVERFLOW} (Event lost or discarded)
 *      2. MODIFY   - {@link StandardWatchEventKinds#ENTRY_MODIFY} (An entry was modified)
 *      3. CREATE   - {@link StandardWatchEventKinds#ENTRY_CREATE} (An entry was created)
 *      4. DELETE   - {@link StandardWatchEventKinds#ENTRY_DELETE} (An entry was deleted)
 * </pre>
 *
 * @author Kimi Liu
 * @since Java 17+
 */
public enum WatchKind {

    /**
     * Indicates that events might have been lost or discarded. This event is not generated by the file system but by
     * the {@link java.nio.file.WatchService} itself when events are missed.
     */
    OVERFLOW(StandardWatchEventKinds.OVERFLOW),
    /**
     * Indicates that a file system entry (a file or a directory) has been modified.
     */
    MODIFY(StandardWatchEventKinds.ENTRY_MODIFY),
    /**
     * Indicates that a file system entry (a file or a directory) has been created.
     */
    CREATE(StandardWatchEventKinds.ENTRY_CREATE),
    /**
     * Indicates that a file system entry (a file or a directory) has been deleted.
     */
    DELETE(StandardWatchEventKinds.ENTRY_DELETE);

    /**
     * An array containing all standard watch event kinds that can be monitored. This includes {@link #OVERFLOW},
     * {@link #MODIFY}, {@link #CREATE}, and {@link #DELETE}.
     */
    public static final WatchEvent.Kind<?>[] ALL = { //
            OVERFLOW.getValue(), // Event lost or discarded
            MODIFY.getValue(), // Entry modified
            CREATE.getValue(), // Entry created
            DELETE.getValue() // Entry deleted
    };

    /**
     * The underlying {@link WatchEvent.Kind} value associated with this enum constant.
     */
    private final WatchEvent.Kind<?> value;

    /**
     * Constructs a {@code WatchKind} enum constant with the specified {@link WatchEvent.Kind} value.
     *
     * @param value The {@link WatchEvent.Kind} value.
     */
    WatchKind(final WatchEvent.Kind<?> value) {
        this.value = value;
    }

    /**
     * Returns the underlying {@link WatchEvent.Kind} value associated with this enum constant.
     *
     * @return The {@link WatchEvent.Kind} value.
     */
    public WatchEvent.Kind<?> getValue() {
        return this.value;
    }

}
