/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.core.lang.range;

import java.io.Serial;
import java.io.Serializable;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.miaixz.bus.core.lang.Assert;

/**
 * Range generator. Generates a stepped list based on given start value, end value, and a stepper. Since the user
 * implements {@link Stepper} to define the stepping logic, the Range itself cannot determine the boundary (whether the
 * end has been reached). The boundary determination logic must be implemented within the {@code step} method.
 *
 * <p>
 * This class uses {@link ReentrantReadWriteLock} to ensure thread safety.
 *
 * @param <T> the type of objects generated by this range
 * @author Kimi Liu
 * @since Java 17+
 */
public class Range<T> implements Iterable<T>, Serializable {

    @Serial
    private static final long serialVersionUID = 2852275092763L;
    /**
     * The starting object of the range.
     */
    private final T start;
    /**
     * The ending object of the range.
     */
    private final T end;

    /**
     * The stepper defines how to increment or decrement the objects in the range.
     */
    private final Stepper<T> stepper;
    /**
     * Indicates whether the starting element is included in the range.
     */
    private final boolean includeStart;
    /**
     * Indicates whether the ending element is included in the range.
     */
    private final boolean includeEnd;

    /**
     * Constructs a new {@code Range} with a specified start object and stepper. The end object is {@code null}, and
     * both start and end are included by default.
     *
     * @param start   the starting object (inclusive)
     * @param stepper the stepper defining the step logic
     */
    public Range(final T start, final Stepper<T> stepper) {
        this(start, null, stepper);
    }

    /**
     * Constructs a new {@code Range} with specified start and end objects, and a stepper. Both start and end are
     * included by default.
     *
     * @param start   the starting object (inclusive)
     * @param end     the ending object (inclusive)
     * @param stepper the stepper defining the step logic
     */
    public Range(final T start, final T end, final Stepper<T> stepper) {
        this(start, end, stepper, true, true);
    }

    /**
     * Constructs a new {@code Range} with full control over start, end, stepper, and inclusion flags.
     *
     * @param start          the starting object
     * @param end            the ending object
     * @param stepper        the stepper defining the step logic
     * @param isIncludeStart {@code true} if the starting element should be included, {@code false} otherwise
     * @param isIncludeEnd   {@code true} if the ending element should be included, {@code false} otherwise
     * @throws IllegalArgumentException if the start object is {@code null}
     */
    public Range(final T start, final T end, final Stepper<T> stepper, final boolean isIncludeStart,
            final boolean isIncludeEnd) {
        Assert.notNull(start, "First element must be not null!");
        this.start = start;
        this.end = end;
        this.stepper = stepper;
        this.includeStart = isIncludeStart;
        this.includeEnd = isIncludeEnd;
    }

    /**
     * Returns an iterator over elements of type {@code T}.
     *
     * @return an {@link Iterator} that iterates through the range
     */
    @Override
    public Iterator<T> iterator() {
        return new Iterator<>() {

            /**
             * The current index in the iteration, starting from 0.
             */
            private int index = 0;
            /**
             * The next object to be returned by the iterator.
             */
            private T next = safeStep(start);

            /**
             * Checks if there are more elements in the range to iterate over.
             *
             * @return {@code true} if there are more elements, {@code false} otherwise
             */
            @Override
            public boolean hasNext() {
                if (0 == this.index && includeStart) {
                    return true;
                }
                if (null == this.next) {
                    return false;
                } else {
                    return includeEnd || !this.next.equals(end);
                }
            }

            /**
             * Returns the next element in the iteration.
             *
             * @return the next element in the range
             * @throws NoSuchElementException if the iteration has no more elements
             */
            @Override
            public T next() {
                if (!this.hasNext()) {
                    throw new NoSuchElementException("Has no next range!");
                }
                return nextUncheck();
            }

            /**
             * This operation is not supported for {@code Range} iterators.
             *
             * @throws UnsupportedOperationException always
             */
            @Override
            public void remove() {
                throw new UnsupportedOperationException("Can not remove ranged element!");
            }

            /**
             * Retrieves the next element without checking {@code hasNext()} and prepares the subsequent element.
             *
             * @return the current element to be returned
             */
            private T nextUncheck() {
                final T current;
                if (0 == this.index) {
                    current = start;
                    if (!includeStart) {
                        // Get the next set of elements
                        index++;
                        return nextUncheck();
                    }
                } else {
                    current = next;
                    this.next = safeStep(this.next);
                }

                index++;
                return current;
            }

            /**
             * Safely performs a step operation without throwing exceptions. If the step fails or results in null, it
             * returns {@code null}.
             *
             * @param base the previous element
             * @return the next stepped element, or {@code null} if stepping fails or ends
             */
            private T safeStep(final T base) {
                final int index = this.index;
                T next = null;
                try {
                    next = stepper.step(base, end, index);
                } catch (final Exception e) {
                    // ignore
                }

                return next;
            }
        };
    }

    /**
     * Returns a string representation of this range, showing its start and end values.
     *
     * @return a string in the format "Range [start=..., end=...]"
     */
    @Override
    public String toString() {
        return "Range [start=" + start + ", end=" + end + "]";
    }

    /**
     * Stepper interface. This interface is used to define how an object should be incremented or decremented in steps.
     * The stepper can define the following logic:
     *
     * <pre>
     * 1. Stepping rules: how the object should be stepped.
     * 2. Step size: by implementing this interface and defining an object property in the implementation class,
     *    the step size can be flexibly defined.
     * 3. Limit on the number of ranges: by implementing this interface and defining an object property in the implementation class,
     *    a limit can be flexibly defined to restrict the number of ranges.
     * </pre>
     *
     * @param <T> the type of object to be stepped
     */
    public interface Stepper<T> {

        /**
         * Increments the object by one step. If the return value after stepping is {@code null}, it indicates that the
         * stepping has ended. The user must define the boundary logic based on the {@code end} parameter; when the
         * boundary is reached, return {@code null} to signify completion. Otherwise, the boundary object in
         * {@code Range} will be ineffective, potentially leading to an infinite loop.
         *
         * @param current the base object from the previous step
         * @param end     the ending object of the range
         * @param index   the current index (which element in the step sequence), starting from 0
         * @return the object after incrementing by one step, or {@code null} if the stepping has ended
         */
        T step(T current, T end, int index);
    }

}
