/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.core.lang.reflect;

import java.lang.reflect.Member;

import org.miaixz.bus.core.lang.Assert;

/**
 * Represents a class member, providing access to class modifiers and other class-level information.
 * <p>
 * Example usage:
 * 
 * <pre>
 * ClassMember member = new ClassMember(String.class);
 * System.out.println(member.getModifiers());
 * </pre>
 *
 * @author Kimi Liu
 * @since Java 17+
 */
public class ClassMember implements Member {

    /**
     * The class represented by this {@code ClassMember} instance.
     */
    private final Class<?> clazz;

    /**
     * Constructs a new {@code ClassMember} instance for the given class.
     *
     * @param clazz The class to be represented. Must not be {@code null}.
     * @throws IllegalArgumentException if {@code clazz} is {@code null}.
     */
    public ClassMember(final Class<?> clazz) {
        this.clazz = Assert.notNull(clazz);
    }

    /**
     * Creates a new {@code ClassMember} instance for the given class using a static factory method.
     *
     * @param clazz The class to be represented. Must not be {@code null}.
     * @return A new {@code ClassMember} object.
     * @throws IllegalArgumentException if {@code clazz} is {@code null}.
     */
    public static ClassMember of(final Class<?> clazz) {
        return new ClassMember(clazz);
    }

    /**
     * Returns the {@code Class} object representing the class that declares this member. For a {@code ClassMember},
     * this is the class itself.
     *
     * @return The {@code Class} object representing the declaring class.
     */
    @Override
    public Class<?> getDeclaringClass() {
        return this.clazz;
    }

    /**
     * Returns the simple name of the underlying class represented by this {@code ClassMember}.
     *
     * @return The simple name of the class.
     */
    @Override
    public String getName() {
        return this.clazz.getName();
    }

    /**
     * Returns the Java language modifiers for the class represented by this {@code ClassMember}. The modifiers are
     * encoded in an integer, as specified by the {@link java.lang.reflect.Modifier} class.
     *
     * @return The integer representing the modifiers for this class.
     */
    @Override
    public int getModifiers() {
        return this.clazz.getModifiers();
    }

    /**
     * Determines if this class is a synthetic class. A synthetic class is one that is generated by the compiler and
     * does not appear in the source code.
     *
     * @return {@code true} if this class is synthetic; {@code false} otherwise.
     */
    @Override
    public boolean isSynthetic() {
        return this.clazz.isSynthetic();
    }

}
