/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.core.lang.selector;

import java.io.Serial;
import java.util.ArrayList;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * A simple round-robin selector.
 *
 * @param <T> the type of the elements
 * @author Kimi Liu
 * @since Java 17+
 */
public class IncrementSelector<T> extends ArrayList<T> implements Selector<T> {

    @Serial
    private static final long serialVersionUID = 2852277613270L;

    /**
     * The current position in the list.
     */
    private final AtomicInteger position = new AtomicInteger();

    /**
     * Constructs an empty {@code IncrementSelector}.
     */
    public IncrementSelector() {
        super();
    }

    /**
     * Constructs a new {@code IncrementSelector} and initializes it with the elements from the given iterable.
     *
     * @param objList the iterable of objects to add
     */
    public IncrementSelector(final Iterable<T> objList) {
        this();
        for (final T object : objList) {
            add(object);
        }
    }

    /**
     * Selects the next element in a round-robin fashion.
     *
     * @return the next element in the sequence
     * @throws IndexOutOfBoundsException if the selector is empty
     */
    @Override
    public T select() {
        final int size = size();
        if (size == 0) {
            throw new IndexOutOfBoundsException("Cannot select from an empty list.");
        }
        return get(position.getAndIncrement() % size);
    }

}
