/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.core.lang.thread;

import java.lang.Thread.UncaughtExceptionHandler;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicInteger;

import org.miaixz.bus.core.lang.Symbol;
import org.miaixz.bus.core.lang.thread.threadlocal.SpecificThread;
import org.miaixz.bus.core.xyz.ClassKit;
import org.miaixz.bus.core.xyz.ObjectKit;
import org.miaixz.bus.core.xyz.StringKit;

/**
 * A {@link ThreadFactory} implementation that provides customizable thread creation. This factory allows for:
 * <ol>
 * <li>Custom thread naming prefixes.</li>
 * <li>Configuration of whether threads are daemon threads.</li>
 * <li>Setting thread priority.</li>
 * <li>Specifying an {@link UncaughtExceptionHandler} for threads.</li>
 * </ol>
 *
 * @author Kimi Liu
 * @since Java 17+
 */
public class NamedThreadFactory implements ThreadFactory {

    /**
     * The prefix for thread names generated by this factory.
     */
    protected final String prefix;
    /**
     * The {@link ThreadGroup} to which new threads will belong.
     */
    protected final ThreadGroup group;
    /**
     * A boolean indicating whether new threads should be daemon threads.
     */
    protected final boolean daemon;
    /**
     * The priority for new threads.
     */
    protected final int priority;
    /**
     * An atomic integer used to generate unique IDs for threads within this factory.
     */
    protected final AtomicInteger nextId = new AtomicInteger(1);
    /**
     * The {@link UncaughtExceptionHandler} for threads created by this factory.
     */
    protected final UncaughtExceptionHandler handler;

    /**
     * Constructs a new {@code NamedThreadFactory} with default settings. Threads will have no specific prefix, belong
     * to the current thread's group, be non-daemon, have normal priority, and no custom exception handler.
     */
    public NamedThreadFactory() {
        this(null, null, Thread.NORM_PRIORITY);
    }

    /**
     * Constructs a new {@code NamedThreadFactory} with a prefix derived from the given class name. Threads will belong
     * to the current thread's group, be non-daemon, have normal priority, and no custom exception handler.
     *
     * @param clazz The class whose simple name will be used as the thread name prefix.
     */
    public NamedThreadFactory(Class<?> clazz) {
        this(clazz, null, Thread.NORM_PRIORITY);
    }

    /**
     * Constructs a new {@code NamedThreadFactory} with a specified thread name prefix. Threads will belong to the
     * current thread's group, be non-daemon, have normal priority, and no custom exception handler.
     *
     * @param prefix The prefix for thread names.
     */
    public NamedThreadFactory(String prefix) {
        this(null, prefix, Thread.NORM_PRIORITY);
    }

    /**
     * Constructs a new {@code NamedThreadFactory} with a prefix derived from the given class name and a specified
     * thread priority. Threads will belong to the current thread's group, be non-daemon, and no custom exception
     * handler.
     *
     * @param clazz    The class whose simple name will be used as the thread name prefix.
     * @param prefix   The prefix for thread names. If {@code null}, the class name will be used.
     * @param priority The priority for new threads (e.g., {@link Thread#NORM_PRIORITY}).
     */
    public NamedThreadFactory(final Class<?> clazz, String prefix, int priority) {
        this(clazz, prefix, true, priority);
    }

    /**
     * Constructs a new {@code NamedThreadFactory} with a specified thread name prefix and daemon status. Threads will
     * belong to the current thread's group, have normal priority, and no custom exception handler.
     *
     * @param prefix The prefix for thread names.
     * @param daemon {@code true} if new threads should be daemon threads; {@code false} otherwise.
     */
    public NamedThreadFactory(final String prefix, final boolean daemon) {
        this(null, prefix, null, daemon, Thread.NORM_PRIORITY);
    }

    /**
     * Constructs a new {@code NamedThreadFactory} with a prefix derived from the given class name, a specified daemon
     * status, and thread priority. Threads will belong to the current thread's group and no custom exception handler.
     *
     * @param clazz    The class whose simple name will be used as the thread name prefix.
     * @param prefix   The prefix for thread names. If {@code null}, the class name will be used.
     * @param daemon   {@code true} if new threads should be daemon threads; {@code false} otherwise.
     * @param priority The priority for new threads (e.g., {@link Thread#NORM_PRIORITY}).
     */
    public NamedThreadFactory(final Class<?> clazz, final String prefix, final boolean daemon, int priority) {
        this(clazz, prefix, null, daemon, priority);
    }

    /**
     * Constructs a new {@code NamedThreadFactory} with a specified thread name prefix, thread group, and daemon status.
     * Threads will have normal priority and no custom exception handler.
     *
     * @param prefix The prefix for thread names.
     * @param group  The {@link ThreadGroup} for new threads. If {@code null}, the current thread's group is used.
     * @param daemon {@code true} if new threads should be daemon threads; {@code false} otherwise.
     */
    public NamedThreadFactory(final String prefix, final ThreadGroup group, final boolean daemon) {
        this(null, prefix, group, daemon, Thread.NORM_PRIORITY, null);
    }

    /**
     * Constructs a new {@code NamedThreadFactory} with a prefix derived from the given class name, a specified thread
     * group, daemon status, and thread priority. Threads will have no custom exception handler.
     *
     * @param clazz    The class whose simple name will be used as the thread name prefix.
     * @param prefix   The prefix for thread names. If {@code null}, the class name will be used.
     * @param group    The {@link ThreadGroup} for new threads. If {@code null}, the current thread's group is used.
     * @param daemon   {@code true} if new threads should be daemon threads; {@code false} otherwise.
     * @param priority The priority for new threads (e.g., {@link Thread#NORM_PRIORITY}).
     */
    public NamedThreadFactory(final Class<?> clazz, final String prefix, final ThreadGroup group, final boolean daemon,
            int priority) {
        this(clazz, prefix, group, daemon, priority, null);
    }

    /**
     * Constructs a new {@code NamedThreadFactory} with a specified thread name prefix, thread group, daemon status, and
     * an {@link UncaughtExceptionHandler}. Threads will have normal priority.
     *
     * @param prefix  The prefix for thread names.
     * @param group   The {@link ThreadGroup} for new threads. If {@code null}, the current thread's group is used.
     * @param daemon  {@code true} if new threads should be daemon threads; {@code false} otherwise.
     * @param handler The {@link UncaughtExceptionHandler} for threads created by this factory.
     */
    public NamedThreadFactory(final String prefix, ThreadGroup group, final boolean daemon,
            final UncaughtExceptionHandler handler) {
        this(null, prefix, group, daemon, Thread.NORM_PRIORITY, handler);
    }

    /**
     * Constructs a new {@code NamedThreadFactory} with full customization options.
     *
     * @param clazz    The class whose simple name will be used as the thread name prefix if {@code prefix} is
     *                 {@code null} or blank.
     * @param prefix   The prefix for thread names. If {@code null} or blank, the class name will be used.
     * @param group    The {@link ThreadGroup} for new threads. If {@code null}, the current thread's group is used.
     * @param daemon   {@code true} if new threads should be daemon threads; {@code false} otherwise.
     * @param priority The priority for new threads (e.g., {@link Thread#NORM_PRIORITY}).
     * @param handler  The {@link UncaughtExceptionHandler} for threads created by this factory.
     * @throws IllegalArgumentException if the provided priority is outside the valid range ({@link Thread#MIN_PRIORITY}
     *                                  to {@link Thread#MAX_PRIORITY}).
     */
    public NamedThreadFactory(final Class<?> clazz, final String prefix, ThreadGroup group, final boolean daemon,
            int priority, final UncaughtExceptionHandler handler) {
        if (priority < Thread.MIN_PRIORITY || priority > Thread.MAX_PRIORITY) {
            throw new IllegalArgumentException(
                    "priority: " + priority + " (expected: Thread.MIN_PRIORITY <= priority <= Thread.MAX_PRIORITY)");
        }

        if (ObjectKit.isEmpty(clazz) && StringKit.isEmpty(prefix)) {
            // If both class and prefix are empty, use a default prefix.
            this.prefix = Symbol.X;
        } else {
            // Use the provided prefix if not blank, otherwise use the class name.
            this.prefix = StringKit.isBlank(prefix) ? ClassKit.getClassName(clazz, true) : prefix;
        }

        if (null == group) {
            // Use the current thread's group if no group is specified.
            group = Thread.currentThread().getThreadGroup();
        }
        this.group = group;
        this.daemon = daemon;
        this.priority = priority;
        this.handler = handler;
    }

    /**
     * Creates a new unstarted {@link Thread} to run the given {@link Runnable}. The thread will be named using the
     * factory's prefix and an incrementing ID. Its daemon status and priority will be set according to the factory's
     * configuration.
     *
     * @param r The {@link Runnable} to be executed by the new thread instance.
     * @return A newly created {@link Thread} instance.
     */
    @Override
    public Thread newThread(final Runnable r) {
        final Thread t = this.newThread(r, StringKit.format("{}{}", this.prefix, this.nextId.getAndIncrement()));
        // Set daemon status if different from the default.
        if (!t.isDaemon()) {
            if (this.daemon) {
                // If the original thread is not daemon, set it to daemon.
                t.setDaemon(true);
            }
        } else if (!this.daemon) {
            // If the original thread is daemon, revert it to non-daemon.
            t.setDaemon(false);
        }
        // Set thread priority if different from the default.
        if (t.getPriority() != this.priority) {
            t.setPriority(this.priority);
        }

        // Set uncaught exception handler if provided.
        if (null != this.handler) {
            t.setUncaughtExceptionHandler(this.handler);
        }
        return t;
    }

    /**
     * Creates a new unstarted {@link Thread} to run the given {@link Runnable} with a specified name.
     *
     * @param r    The {@link Runnable} to be executed by the new thread instance.
     * @param name The name for the new thread.
     * @return A newly created {@link Thread} instance.
     */
    protected Thread newThread(Runnable r, String name) {
        return new SpecificThread(this.group, r, name);
    }

}
