/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.core.text.dfa;

import java.util.Set;

import org.miaixz.bus.core.xyz.SetKit;

/**
 * Utility class for handling stop characters, such as punctuation and whitespace, which are typically ignored during
 * text processing, especially in DFA-based algorithms.
 *
 * @author Kimi Liu
 * @since Java 17+
 */
public class StopChar {

    /**
     * A set of characters that are considered stop words or characters that do not require processing. This includes
     * various punctuation marks, symbols, and whitespace characters.
     */
    public static final Set<Character> STOP_WORD = SetKit.of(' ', '\'', '、', '。', '·', 'ˉ', 'ˇ', '々', '—', '～', '‖',
            '…', '‘', '’', '“', '”', '〔', '〕', '〈', '〉', '《', '》', '「', '」', '『', '』', '〖', '〗', '【', '】', '±', '＋',
            '－', '×', '÷', '∧', '∨', '∑', '∏', '∪', '∩', '∈', '√', '⊥', '⊙', '∫', '∮', '≡', '≌', '≈', '∽', '∝', '≠',
            '≮', '≯', '≤', '≥', '∞', '∶', '∵', '∴', '∷', '♂', '♀', '°', '′', '〃', '℃', '＄', '¤', '￠', '￡', '‰', '§',
            '☆', '★', '〇', '○', '●', '◎', '◇', '◆', '□', '■', '△', '▽', '⊿', '▲', '▼', '◣', '◤', '◢', '◥', '▁', '▂',
            '▃', '▄', '▅', '▆', '▇', '█', '▉', '▊', '▋', '▌', '▍', '▎', '▏', '▓', '※', '→', '←', '↑', '↓', '↖', '↗',
            '↘', '↙', '〓', 'ⅰ', 'ⅱ', 'ⅲ', 'ⅳ', 'ⅴ', 'ⅵ', 'ⅶ', 'ⅷ', 'ⅸ', 'ⅹ', '①', '②', '③', '④', '⑤', '⑥', '⑦', '⑧',
            '⑨', '⑩', '⒈', '⒉', '⒊', '⒋', '⒌', '⒍', '⒎', '⒏', '⒐', '⒑', '⒒', '⒓', '⒔', '⒕', '⒖', '⒗', '⒘', '⒙', '⒚',
            '⒛', '⑴', '⑵', '⑶', '⑷', '⑸', '⑹', '⑺', '⑻', '⑼', '⑽', '⑾', '⑿', '⒀', '⒁', '⒂', '⒃', '⒄', '⒅', '⒆', '⒇',
            'Ⅰ', 'Ⅱ', 'Ⅲ', 'Ⅳ', 'Ⅴ', 'Ⅵ', 'Ⅶ', 'Ⅷ', 'Ⅸ', 'Ⅹ', 'Ⅺ', 'Ⅻ', '！', '＃', '￥', '％', '＆', '（', '）', '＊', '，',
            '．', '／', '０', '１', '２', '３', '４', '５', '６', '７', '８', '９', '：', '；', '＜', '＝', '＞', '？', '＠', '〔', '＼',
            '〕', '＾', '＿', '‘', '｛', '｜', '｝', 'Ρ', 'Υ', 'Φ', 'Χ', 'Ψ', 'Ω', 'α', 'β', 'γ', 'δ', 'ε', 'ζ', 'η', 'θ',
            'ι', 'κ', 'λ', 'μ', 'ν', 'ξ', 'ο', 'π', 'ρ', 'σ', 'τ', 'υ', 'φ', 'χ', 'ψ', 'ω', '（', '）', '＾', '﹊', '﹍',
            '╭', '╮', '╰', '╯', '', '_', '', '^', '（', '：', '！', '/', '\\', '\"', '<', '>', '`', '{', '}', '~', '(',
            ')', '-', '$', '@', '*', '&', '#', '卐', '㎎', '㎏', '㎜', '㎝', '㎞', '㎡', '㏄', '㏎', '㏑', '㏒', '㏕', '+', '=',
            '?', ':', '.', '!', ';', ']', '|', '%');

    /**
     * Checks if the given character is a stop character. Stop characters are typically ignored during text processing.
     *
     * @param ch The character to check.
     * @return {@code true} if the character is a whitespace character or present in the {@link #STOP_WORD} set,
     *         {@code false} otherwise.
     */
    public static boolean isStopChar(final char ch) {
        return Character.isWhitespace(ch) || STOP_WORD.contains(ch);
    }

    /**
     * Checks if the given character is not a stop character (i.e., it is a valid character for processing).
     *
     * @param ch The character to check.
     * @return {@code true} if the character is not a stop character, {@code false} otherwise.
     */
    public static boolean isNotStopChar(final char ch) {
        return !isStopChar(ch);
    }

}
