/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.core.text.escape;

import java.io.Serial;

import org.miaixz.bus.core.text.replacer.LookupReplacer;
import org.miaixz.bus.core.text.replacer.ReplacerChain;

/**
 * Unescapes special XML characters. This class provides functionality to convert XML entity references back into their
 * original characters.
 *
 * <p>
 * It unescapes the following XML entities:
 *
 * <ul>
 * <li>{@code &amp;} to {@code &} (ampersand)</li>
 * <li>{@code &lt;} to {@code <} (less than)</li>
 * <li>{@code &gt;} to {@code >} (greater than)</li>
 * <li>{@code &quot;} to {@code "} (double quote)</li>
 * <li>{@code &apos;} to {@code '} (single quote)</li>
 * </ul>
 * It also handles numeric entity unescaping.
 *
 * @author Kimi Liu
 * @since Java 17+
 */
public class XmlUnescape extends ReplacerChain {

    /**
     * Basic XML unescape lookup table. This array is generated by inverting the {@link XmlEscape#BASIC_ESCAPE} array,
     * mapping XML entity references back to their original characters.
     */
    protected static final String[][] BASIC_UNESCAPE = invert(XmlEscape.BASIC_ESCAPE);
    /**
     * The serial version UID for serialization.
     */
    @Serial
    private static final long serialVersionUID = 2852236398787L;

    /**
     * Constructs a new {@code XmlUnescape} instance. Initializes the unescape chain with the basic XML unescape lookup
     * replacer and a {@link NumericEntityUnescaper} for handling numeric character references.
     */
    public XmlUnescape() {
        addChain(new LookupReplacer(BASIC_UNESCAPE));
        addChain(new NumericEntityUnescaper());
    }

    /**
     * Inverts a 2D string array by swapping the elements at index 0 and 1 within each inner array. This is typically
     * used to create an unescape lookup table from an escape lookup table.
     *
     * @param array The 2D string array to be inverted. Each inner array is expected to have at least two elements.
     * @return A new 2D string array where the first and second elements of each inner array are swapped.
     */
    public static String[][] invert(final String[][] array) {
        final String[][] newarray = new String[array.length][2];
        for (int i = 0; i < array.length; i++) {
            newarray[i][0] = array[i][1];
            newarray[i][1] = array[i][0];
        }
        return newarray;
    }

}
