/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.crypto.builtin;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;

import org.miaixz.bus.core.lang.Assert;
import org.miaixz.bus.core.lang.Charset;
import org.miaixz.bus.core.lang.exception.InternalException;
import org.miaixz.bus.core.xyz.ArrayKit;
import org.miaixz.bus.core.xyz.ByteKit;
import org.miaixz.bus.core.xyz.StringKit;

/**
 * Utility class for handling salt magic values, typically used to extract salt information from ciphertext generated by
 * OpenSSL.
 *
 * @author Kimi Liu
 * @since Java 17+
 */
public class SaltMagic {

    /**
     * The length of the salt value in bytes.
     */
    public static final byte SALT_LEN = 8;
    /**
     * OpenSSL's magic initial bytes, indicating salted encryption.
     */
    public static final byte[] SALTED_MAGIC = "Salted__".getBytes(Charset.US_ASCII);

    /**
     * The combined length of the magic bytes and the salt, which is 16 bytes (128 bits).
     */
    public static final int MAGIC_SALT_LENGTH = SALTED_MAGIC.length + SALT_LEN;

    /**
     * Retrieves the actual encrypted data by removing the leading salt header, if present. If the data does not start
     * with the {@link #SALTED_MAGIC} bytes, the original data is returned.
     *
     * @param encryptedData The ciphertext, potentially including the salt header.
     * @return The actual ciphertext without the salt header.
     */
    public static byte[] getData(final byte[] encryptedData) {
        if (ArrayKit.startWith(encryptedData, SALTED_MAGIC)) {
            return Arrays.copyOfRange(encryptedData, SALTED_MAGIC.length + SALT_LEN, encryptedData.length);
        }
        return encryptedData;
    }

    /**
     * Retrieves the salt value from an input stream without closing the stream. The stream is expected to start with
     * the {@link #SALTED_MAGIC} bytes followed by the salt.
     *
     * @param in The {@link InputStream} containing the encrypted data.
     * @return The extracted salt as a byte array.
     * @throws InternalException if an I/O error occurs, or if the magic header or salt is unexpected/incomplete.
     */
    public static byte[] getSalt(final InputStream in) throws InternalException {
        final byte[] headerBytes = new byte[SALTED_MAGIC.length];

        try {
            final int readHeaderSize = in.read(headerBytes);
            if (readHeaderSize < headerBytes.length || !Arrays.equals(SALTED_MAGIC, headerBytes)) {
                throw new InternalException("Unexpected magic header " + StringKit.toString(headerBytes));
            }

            final byte[] salt = new byte[SALT_LEN];
            final int readSaltSize = in.read(salt);
            if (readSaltSize < salt.length) {
                throw new InternalException("Unexpected salt: " + StringKit.toString(salt));
            }
            return salt;
        } catch (final IOException e) {
            throw new InternalException(e);
        }
    }

    /**
     * Retrieves the 8-byte salt random number from encrypted data. This method assumes the encrypted data starts with
     * {@link #SALTED_MAGIC} followed by the salt.
     *
     * @param encryptedData The ciphertext, potentially including the salt header.
     * @return The 8-byte salt random number, or {@code null} if the data does not start with the magic header.
     */
    public static byte[] getSalt(final byte[] encryptedData) {
        if (ArrayKit.startWith(encryptedData, SALTED_MAGIC)) {
            return Arrays.copyOfRange(encryptedData, SALTED_MAGIC.length, MAGIC_SALT_LENGTH);
        }
        return null;
    }

    /**
     * Adds a magic header to the encrypted data. The generated ciphertext format is:
     * 
     * <pre>
     * Salted__[salt][data]
     * </pre>
     *
     * @param data The encrypted data.
     * @param salt The salt value, which must be 8 bytes long. If {@code null}, the original data is returned without
     *             modification.
     * @return The ciphertext with the magic header and salt prepended.
     * @throws IllegalArgumentException if the salt length is not 8 bytes.
     */
    public static byte[] addMagic(final byte[] data, final byte[] salt) {
        if (null == salt) {
            return data;
        }
        Assert.isTrue(SALT_LEN == salt.length);
        return ByteKit.concat(SALTED_MAGIC, salt, data);
    }

    /**
     * Retrieves the magic header, which includes the "Salted__" prefix and the salt value. The generated format is:
     * 
     * <pre>
     * Salted__[salt]
     * </pre>
     *
     * @param salt The salt value, which must be 8 bytes long and not {@code null}.
     * @return The magic header as a byte array.
     * @throws IllegalArgumentException if the salt is null or its length is not 8 bytes.
     */
    public static byte[] getSaltedMagic(final byte[] salt) {
        return ByteKit.concat(SALTED_MAGIC, salt);
    }

}
