/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.crypto.builtin.digest.mac;

import java.io.IOException;
import java.io.InputStream;

import org.miaixz.bus.core.lang.Normal;
import org.miaixz.bus.core.lang.exception.CryptoException;

/**
 * MAC (Message Authentication Code) algorithm engine interface. This interface defines the contract for MAC algorithm
 * implementations, providing methods to update the MAC with data, finalize the MAC calculation, reset the MAC, and
 * retrieve properties like MAC length and algorithm name.
 * <p>
 * Note: Implementations of this interface may not be thread-safe.
 * </p>
 *
 * @author Kimi Liu
 * @since Java 17+
 */
public interface Mac {

    /**
     * Updates the MAC with the entire contents of the given byte array.
     *
     * @param in The byte array containing the data to be processed.
     */
    default void update(final byte[] in) {
        update(in, 0, in.length);
    }

    /**
     * Updates the MAC with the specified portion of the given byte array.
     *
     * @param in    The byte array containing the data to be processed.
     * @param inOff The offset in the input array where the data begins.
     * @param len   The number of bytes to process from the input array.
     */
    void update(byte[] in, int inOff, int len);

    /**
     * Completes the MAC calculation and returns the result. The MAC is reset after this operation.
     *
     * @return The calculated MAC value as a byte array.
     */
    byte[] doFinal();

    /**
     * Resets the MAC to its initial state, discarding all processed data.
     */
    void reset();

    /**
     * Generates a message digest (MAC) from the data provided by an {@link InputStream}. Uses a default buffer size of
     * {@link Normal#_8192} if {@code bufferLength} is less than 1. The MAC is reset after this operation.
     *
     * @param data         The {@link InputStream} containing the data to be digested.
     * @param bufferLength The buffer length to use for reading the stream. If less than 1, {@link Normal#_8192} is used
     *                     as default.
     * @return The calculated MAC value as a byte array.
     * @throws CryptoException if an I/O error occurs during stream reading.
     */
    default byte[] digest(final InputStream data, int bufferLength) {
        if (bufferLength < 1) {
            bufferLength = Normal._8192;
        }

        final byte[] buffer = new byte[bufferLength];

        byte[] result;
        try {
            int read = data.read(buffer, 0, bufferLength);

            while (read > -1) {
                update(buffer, 0, read);
                read = data.read(buffer, 0, bufferLength);
            }
            result = doFinal();
        } catch (final IOException e) {
            throw new CryptoException(e);
        } finally {
            reset();
        }
        return result;
    }

    /**
     * Retrieves the length of the MAC in bytes.
     *
     * @return The MAC length in bytes.
     */
    int getMacLength();

    /**
     * Retrieves the name of the MAC algorithm.
     *
     * @return The algorithm name.
     */
    String getAlgorithm();

}
