/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.image.galaxy.io;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.spi.FileTypeDetector;

import org.miaixz.bus.core.lang.MediaType;
import org.miaixz.bus.core.xyz.StreamKit;

/**
 * A {@link FileTypeDetector} implementation that detects Bzip2 compressed files. It checks for the Bzip2 magic number
 * (0x42 0x5A 0x68) at the beginning of the file. It also handles specific file extensions for VCF Bzip2 files.
 * 
 * @author Kimi Liu
 * @since Java 17+
 */
public class Bzip2FileDetector extends FileTypeDetector {

    /**
     * Checks if the given filename ends with any of the specified extensions (case-insensitive).
     * 
     * @param filename The name of the file.
     * @param exts     An array of extensions to check against.
     * @return {@code true} if the filename ends with one of the extensions, {@code false} otherwise.
     */
    private static boolean endsWith(String filename, String... exts) {
        for (String ext : exts) {
            if (filename.endsWith(ext)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public String probeContentType(Path path) throws IOException {
        byte[] b = new byte[3];
        try (InputStream in = Files.newInputStream(path)) {
            return StreamKit.readAvailable(in, b, 0, 3) == 3 && b[0] == 0x42 && b[1] == 0x5A && b[2] == 0x68
                    ? endsWith(path.getFileName().toString().toLowerCase(), ".vcf.bz2", ".vcfbzip2", ".vcfbz2")
                            ? MediaType.APPLICATION_PRS_VCFBZIP
                            : MediaType.APPLICATION_X_BZIP2
                    : null;
        }
    }

}
