/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.image.plugin;

import org.miaixz.bus.core.xyz.IoKit;
import org.miaixz.bus.image.*;
import org.miaixz.bus.image.galaxy.data.Attributes;
import org.miaixz.bus.image.metric.Connection;
import org.miaixz.bus.image.metric.net.ApplicationEntity;
import org.miaixz.bus.logger.Logger;

import java.text.MessageFormat;

/**
 * The {@code Echo} class provides a simple way to perform a DICOM C-ECHO verification. It encapsulates the setup and
 * execution of the echo process.
 *
 * @author Kimi Liu
 * @since Java 17+
 */
public class Echo {

    /**
     * Performs a DICOM C-ECHO verification.
     *
     * @param callingAET the AET of the calling DICOM node.
     * @param calledNode the configuration of the called DICOM node.
     * @return a {@link Status} instance containing the DICOM response, status, error message, and timing information.
     */
    public static Status process(String callingAET, Node calledNode) {
        return process(new Node(callingAET), calledNode);
    }

    /**
     * Performs a DICOM C-ECHO verification.
     *
     * @param callingNode the configuration of the calling DICOM node.
     * @param calledNode  the configuration of the called DICOM node.
     * @return a {@link Status} instance containing the DICOM response, status, error message, and timing information.
     */
    public static Status process(Node callingNode, Node calledNode) {
        return process(null, callingNode, calledNode);
    }

    /**
     * Performs a DICOM C-ECHO verification with advanced options.
     *
     * @param args        optional advanced parameters (proxy, authentication, connection, and TLS).
     * @param callingNode the configuration of the calling DICOM node.
     * @param calledNode  the configuration of the called DICOM node.
     * @return a {@link Status} instance containing the DICOM response, status, error message, and timing information.
     */
    public static Status process(Args args, Node callingNode, Node calledNode) {
        if (callingNode == null || calledNode == null) {
            throw new IllegalArgumentException("callingNode or calledNode cannot be null!");
        }

        Args options = args == null ? new Args() : args;

        try {
            Device device = new Device("storescu");
            Connection conn = new Connection();
            device.addConnection(conn);
            ApplicationEntity ae = new ApplicationEntity(callingNode.getAet());
            device.addApplicationEntity(ae);
            ae.addConnection(conn);
            StoreSCU storeSCU = new StoreSCU(ae, null);
            Connection remote = storeSCU.getRemoteConnection();
            Centre service = new Centre(device);

            options.configureConnect(storeSCU.getAAssociateRQ(), remote, calledNode);
            options.configureBind(ae, conn, callingNode);

            // configure
            options.configure(conn);
            options.configureTLS(conn, remote);

            storeSCU.setPriority(options.getPriority());

            service.start();
            try {
                long t1 = System.currentTimeMillis();
                storeSCU.open();
                long t2 = System.currentTimeMillis();
                Attributes rsp = storeSCU.echo();
                long t3 = System.currentTimeMillis();
                String message = MessageFormat.format(
                        "Successful DICOM Echo. Connected in {2}ms from {0} to {1}. Service execution in {3}ms.",
                        storeSCU.getAAssociateRQ().getCallingAET(),
                        storeSCU.getAAssociateRQ().getCalledAET(),
                        t2 - t1,
                        t3 - t2);
                Status dcmState = new Status(rsp.getInt(Tag.Status, Status.Success), message, null);
                dcmState.addProcessTime(t1, t2, t3);
                return dcmState;
            } finally {
                IoKit.close(storeSCU);
                service.stop();
            }
        } catch (Exception e) {
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            String message = "DICOM Echo failed, storescu: " + e.getMessage();
            Logger.error(message, e);
            return Status.buildMessage(new Status(Status.UnableToProcess, message, null), null, e);
        }
    }

}
