/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.mapper;

import java.lang.reflect.Method;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

import org.apache.ibatis.annotations.Lang;
import org.apache.ibatis.builder.annotation.ProviderContext;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.mapping.SqlSource;
import org.apache.ibatis.scripting.xmltags.XMLLanguageDriver;
import org.apache.ibatis.session.Configuration;
import org.miaixz.bus.core.Context;
import org.miaixz.bus.logger.Logger;
import org.miaixz.bus.mapper.parsing.SqlMetaCache;
import org.miaixz.bus.mapper.parsing.SqlSourceEnhancer;
import org.miaixz.bus.mapper.parsing.TableMeta;

/**
 * A custom MyBatis {@link XMLLanguageDriver} that caches XML-based SqlSource to avoid redundant parsing.
 *
 * @author Kimi Liu
 * @since Java 17+
 */
public class Caching extends XMLLanguageDriver {

    /**
     * A map to cache {@link SqlMetaCache} objects. The initial capacity is set based on an estimate (e.g., 30 entities
     * with 25 methods each).
     * <p>
     * For a single data source, this cache can eventually be cleared. For multiple data sources, the cache must be
     * retained because the cleanup timing is indeterminate.
     * </p>
     */
    private static final Map<String, SqlMetaCache> CACHE_SQL = new ConcurrentHashMap<>(
            Context.INSTANCE.getInt(Args.INITSIZE_KEY, 1024));

    /**
     * Caches {@link SqlSource} per {@link Configuration} to handle multi-datasource or multi-configuration scenarios
     * (e.g., in unit tests), ensuring consistency.
     */
    private static final Map<Configuration, Map<String, SqlSource>> CONFIGURATION_CACHE_KEY_MAP = new ConcurrentHashMap<>(
            4);

    /**
     * If true, the cache is cleared after its first use, allowing for garbage collection. This should be set to
     * {@code false} when using multiple data sources with a single SqlSessionFactory to prevent premature cache
     * eviction. It can be set to {@code true} for a single SqlSessionFactory with multiple data sources. Defaults to
     * {@code false}.
     */
    private static final boolean USE_ONCE = Context.INSTANCE.getBoolean(Args.USEONCE_KEY, false);

    /**
     * Generates a cache key based on the mapper interface and method.
     *
     * @param providerContext The provider context containing mapper interface and method information.
     * @return A cache key, which is interned to be used as a lock object.
     */
    private static String cacheKey(ProviderContext providerContext) {
        return (providerContext.getMapperType().getName() + "." + providerContext.getMapperMethod().getName()).intern();
    }

    /**
     * Checks if the mapper method is annotated with {@code @Lang(Caching.class)}.
     *
     * @param providerContext The provider context containing method information.
     * @throws RuntimeException if the method is not annotated with {@code @Lang(Caching.class)}.
     */
    private static void isAnnotationPresentLang(ProviderContext providerContext) {
        Method mapperMethod = providerContext.getMapperMethod();
        if (mapperMethod.isAnnotationPresent(Lang.class)) {
            Lang lang = mapperMethod.getAnnotation(Lang.class);
            if (lang.value() == Caching.class) {
                return;
            }
        }
        throw new RuntimeException(
                mapperMethod + " need to configure @Lang(Caching.class) to use the Caching.cache method for caching");
    }

    /**
     * Caches the SQL script and its associated metadata.
     *
     * @param providerContext   The provider context, containing mapper interface and method information.
     * @param entity            The entity metadata.
     * @param sqlScriptSupplier A supplier for the SQL script string.
     * @return The generated cache key.
     */
    public static String cache(ProviderContext providerContext, TableMeta entity, Supplier<String> sqlScriptSupplier) {
        String cacheKey = cacheKey(providerContext);
        if (!CACHE_SQL.containsKey(cacheKey)) {
            isAnnotationPresentLang(providerContext);
            synchronized (cacheKey) {
                if (!CACHE_SQL.containsKey(cacheKey)) {
                    CACHE_SQL.put(
                            cacheKey,
                            new SqlMetaCache(Objects.requireNonNull(providerContext), Objects.requireNonNull(entity),
                                    Objects.requireNonNull(sqlScriptSupplier)));
                }
            }
        }
        return cacheKey;
    }

    /**
     * Creates an {@link SqlSource}. If a cached version exists, it is reused; otherwise, a new instance is created and
     * cached. This method uses the script parameter as a key to look up pre-parsed SQL metadata.
     *
     * @param configuration The MyBatis configuration.
     * @param script        The script content or a cache key.
     * @param parameterType The parameter type class.
     * @return The created or cached {@link SqlSource}.
     */
    @Override
    public SqlSource createSqlSource(Configuration configuration, String script, Class<?> parameterType) {
        if (CACHE_SQL.containsKey(script)) {
            String cacheKey = script;
            if (!(CONFIGURATION_CACHE_KEY_MAP.containsKey(configuration)
                    && CONFIGURATION_CACHE_KEY_MAP.get(configuration).containsKey(cacheKey))) {
                synchronized (cacheKey) {
                    if (!(CONFIGURATION_CACHE_KEY_MAP.containsKey(configuration)
                            && CONFIGURATION_CACHE_KEY_MAP.get(configuration).containsKey(cacheKey))) {
                        SqlMetaCache cache = CACHE_SQL.get(cacheKey);
                        if (cache == SqlMetaCache.NULL) {
                            throw new RuntimeException(script
                                    + " => CACHE_SQL is NULL, you need to configure mapper.provider.cacheSql.useOnce=false");
                        }
                        cache.getTableMeta().initRuntimeContext(configuration, cache.getProviderContext(), cacheKey);
                        Map<String, SqlSource> cachekeyMap = CONFIGURATION_CACHE_KEY_MAP
                                .computeIfAbsent(configuration, k -> new ConcurrentHashMap<>());
                        MappedStatement ms = configuration.getMappedStatement(cacheKey);
                        Registry.SPI.customize(cache.getTableMeta(), ms, cache.getProviderContext());
                        String sqlScript = cache.getSqlScript();
                        if (Logger.isTraceEnabled()) {
                            Logger.trace("cacheKey - " + cacheKey + " :\n" + sqlScript + "\n");
                        }
                        SqlSource sqlSource = super.createSqlSource(configuration, sqlScript, parameterType);
                        sqlSource = SqlSourceEnhancer.SPI
                                .customize(sqlSource, cache.getTableMeta(), ms, cache.getProviderContext());
                        cachekeyMap.put(cacheKey, sqlSource);
                        if (USE_ONCE) {
                            CACHE_SQL.put(cacheKey, SqlMetaCache.NULL);
                        }
                    }
                }
            }
            return CONFIGURATION_CACHE_KEY_MAP.get(configuration).get(cacheKey);
        } else {
            return super.createSqlSource(configuration, script, parameterType);
        }
    }

}
