/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.spring.boot;

import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;

/**
 * An implementation of {@link BeanPostProcessor} that injects a {@link StartupReporter} into any bean that implements
 * the {@link StartupReporterAware} interface.
 *
 * @author Kimi Liu
 * @since Java 17+
 */
public class StartupReporterProcessor implements BeanPostProcessor {

    /**
     * The {@link StartupReporter} instance to be injected.
     */
    private final StartupReporter startupReporter;

    /**
     * Constructs a new {@code StartupReporterProcessor} with the given {@link StartupReporter}.
     *
     * @param startupReporter The {@link StartupReporter} instance.
     */
    public StartupReporterProcessor(StartupReporter startupReporter) {
        this.startupReporter = startupReporter;
    }

    /**
     * Applies this {@code BeanPostProcessor} to the given new bean instance before any construction callbacks (like
     * {@code InitializingBean}'s {@code afterPropertiesSet} or a custom init-method).
     * <p>
     * If the bean implements {@link StartupReporterAware}, its {@code setStartupReporter} method is called with the
     * {@code startupReporter} instance.
     * </p>
     *
     * @param bean     The new bean instance.
     * @param beanName The name of the bean.
     * @return The given bean instance.
     * @throws BeansException in case of initialization errors.
     */
    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        if (bean instanceof StartupReporterAware) {
            ((StartupReporterAware) bean).setStartupReporter(startupReporter);
        }
        return bean;
    }

}
