/*
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
 ~                                                                               ~
 ~ The MIT License (MIT)                                                         ~
 ~                                                                               ~
 ~ Copyright (c) 2015-2025 miaixz.org and other contributors.                    ~
 ~                                                                               ~
 ~ Permission is hereby granted, free of charge, to any person obtaining a copy  ~
 ~ of this software and associated documentation files (the "Software"), to deal ~
 ~ in the Software without restriction, including without limitation the rights  ~
 ~ to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     ~
 ~ copies of the Software, and to permit persons to whom the Software is         ~
 ~ furnished to do so, subject to the following conditions:                      ~
 ~                                                                               ~
 ~ The above copyright notice and this permission notice shall be included in    ~
 ~ all copies or substantial portions of the Software.                           ~
 ~                                                                               ~
 ~ THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    ~
 ~ IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      ~
 ~ FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   ~
 ~ AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        ~
 ~ LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, ~
 ~ OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     ~
 ~ THE SOFTWARE.                                                                 ~
 ~                                                                               ~
 ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~ ~
*/
package org.miaixz.bus.vortex.registry;

import org.miaixz.bus.vortex.magic.Limiter;

/**
 * Default rate limiter registry implementation, providing registration and management functionalities for rate limiter
 * configurations ({@link Limiter}) based on {@link AbstractRegistry}.
 *
 * @author Justubborn
 * @since Java 17+
 */
public class LimiterRegistry extends AbstractRegistry<Limiter> {

    /**
     * Constructs a {@code LimiterRegistry} and sets the key generation strategy. The key for a limiter is generated by
     * concatenating its IP, method, and version.
     */
    public LimiterRegistry() {
        setKeyGenerator(limiter -> limiter.getIp() + limiter.getMethod() + limiter.getVersion());
    }

    /**
     * Retrieves a rate limiter configuration based on its IP address, method name, and version number.
     *
     * @param ip      The IP address.
     * @param method  The method name.
     * @param version The version number.
     * @return The matching {@link Limiter} object, or {@code null} if no configuration is found.
     */
    public Limiter get(String ip, String method, String version) {
        return this.get(ip + method + version);
    }

    /**
     * Initializes the registry. This is an empty default implementation, intended for subclasses to extend as needed.
     */
    @Override
    public void init() {
        // Empty implementation, for subclasses to extend as needed
    }

}
