/*
Copyright 2016 The Rook Authors. All rights reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

	http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package smoke

import (
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
)

// Smoke Test for ObjectStore - Test check the following operations on ObjectStore in order
//Create object store, Create User, Connect to Object Store, Create Bucket, Read/Write/Delete to bucket,Delete Bucket and
//Delete user
func (suite *SmokeSuite) TestMonFailover() {
	logger.Infof("Mon Failover Smoke Test")

	namespace := "rook"
	opts := metav1.ListOptions{LabelSelector: "app=rook-ceph-mon"}
	monReplicas, err := suite.k8sh.Clientset.ReplicaSets(namespace).List(opts)
	require.Nil(suite.T(), err)
	assert.Equal(suite.T(), 3, len(monReplicas.Items))

	monToKill := monReplicas.Items[0].Name
	logger.Infof("Killing mon %s", monToKill)
	propagation := metav1.DeletePropagationForeground
	delOptions := &metav1.DeleteOptions{PropagationPolicy: &propagation}
	err = suite.k8sh.Clientset.ReplicaSets(namespace).Delete(monToKill, delOptions)
	require.Nil(suite.T(), err)

	// Wait for the health check to start a new monitor
	for i := 0; i < 10; i++ {
		monReplicas, err := suite.k8sh.Clientset.ReplicaSets(namespace).List(opts)
		require.Nil(suite.T(), err)

		// Make sure the old mon is not still alive
		foundOldMon := false
		for _, mon := range monReplicas.Items {
			if mon.Name == monToKill {
				foundOldMon = true
			}
		}

		// Check if we have three monitors
		if !foundOldMon {
			if len(monReplicas.Items) == 3 {
				var newMons []string
				for _, mon := range monReplicas.Items {
					newMons = append(newMons, mon.Name)
				}
				logger.Infof("Found a new monitor! monitors=%v", newMons)
				return
			}

			assert.Equal(suite.T(), 2, len(monReplicas.Items))
		}

		logger.Infof("Waiting for a new monitor to start")
		time.Sleep(8 * time.Second)
	}

	require.Fail(suite.T(), "giving up waiting for a new monitor")
}
