/* agbtty.c
   a tty for the GBA

Copyright 2003 Damian Yerrick

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF
OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
IN THE SOFTWARE.

*/

#include "pin8gba.h"
#include <errno.h>
#include "agbtty.h"
#include "chr.h"

#define DEFAULT_MAP 4

/* which map we're using (0-30) */
static unsigned char nt = DEFAULT_MAP;

/* current scroll position (0-63) */
static unsigned char curs_x, curs_y;
static unsigned char scrollbase;

/* current scrollback position (0-44) */
unsigned int agbtty_cur_scrollback;
static unsigned char show_cursor;

/* current text attribute, in map entry format */
static unsigned short text_attr;



static int agbtty_cls(void)
{
  unsigned int y;
  unsigned int c = text_attr | ' ';

  c |= c << 16;

  for(y = 0; y < 20; y++)
  {
    u32 *base = (u32 *)(MAP[nt][(y + scrollbase) & 63]);
    unsigned int x;

    for(x = 5; x > 0; x--)  /* clear 6 spaces each time through */
    {
      *base++ = c;
      *base++ = c;
      *base++ = c;
    }
  }
  curs_x = 0;
  curs_y = 0;
  agbtty_scrollback(0);
  return 0;
}

/* upcvt_4bit() ************************
   Convert a 1-bit font to GBA 4-bit format.
*/
static void upcvt_4bit(void *dst, const u8 *src, size_t len)
{
  u32 *out = dst;

  for(; len > 0; len--)
  {
    u32 dst_bits = 0;
    u32 src_bits = *src++;
    u32 x;

    for(x = 0; x < 8; x++)
    {
      dst_bits <<= 4;
      dst_bits |= src_bits & 1;
      src_bits >>= 1;
    }
    *out++ = dst_bits;
  }
}


static int move_cursor(void)
{
  if(show_cursor)
  {
    int pixel_y = (curs_y + agbtty_cur_scrollback) * 8;

    if(pixel_y < 160)
    {
      OAM[0].y = pixel_y | OAM_16C | OAM_SQUARE;
      OAM[0].x = (curs_x << 3) | OAM_SIZE0;
    }
    else
    {
      OAM[0].y = OAM_HIDDEN;
    }
  }
  return 0;
}


int agbtty_show_cursor(int shown)
{
  if(shown)
  {
    show_cursor = 1;
    OAM[0].tile = 1023 | OAM_PRI(1);
    move_cursor();
  }
  else
  {
    show_cursor = 0;
    OAM[0].y = OAM_HIDDEN;
  }
  return 0;
}

int agbtty_init_cursor(void)
{
  const u32 cursor_img[8] =
  {
    0x00000000,
    0x00000000,
    0x00000000,
    0x00000000,
    0x00000000,
    0x00000000,
    0x10101010,
    0x11111111
  };
  unsigned int i;

  for(i = 0; i < 128; i++)
  {
    OAM[0].y = OAM_HIDDEN;
  }

  {
    u32 *last_oam_tile = (u32 *)0x06017fe0;

    for(i = 0; i < 8; i++)
      last_oam_tile[i] = cursor_img[i];
  }
  return agbtty_show_cursor(1);
}


static int agbtty_beep(void)
{
  SQR1CTRL = 0xd140;
  SQR1FREQ = 1917 | FREQ_HOLD | FREQ_RESET;
  return 0;
}


static int init_sound(void)
{
  //turn on sound circuit
  SNDSTAT = SNDSTAT_ENABLE;
  //full volume, enable sound 3 to left and right
  DMGSNDCTRL = DMGSNDCTRL_LVOL(7) | DMGSNDCTRL_RVOL(7) |
               DMGSNDCTRL_LSQR1 | DMGSNDCTRL_RSQR1 | 
               DMGSNDCTRL_LSQR2 | DMGSNDCTRL_RSQR2 | 
               DMGSNDCTRL_LTRI | DMGSNDCTRL_RTRI |
               DMGSNDCTRL_LNOISE | DMGSNDCTRL_RNOISE;
  DSOUNDCTRL = 0x0b06;

  SQR1SWEEP = SQR1SWEEP_OFF;
  SETSNDRES(1);

  return 0;
}

int agbtty_init(void)
{
  upcvt_4bit(VRAM, text_chr, text_chr_len);
  agbtty_set_map(DEFAULT_MAP);
  scrollbase = 0;
  agbtty_textattr(0);
  agbtty_cls();
  BGCTRL[0] = 0 | BGCTRL_PAT(0) | BGCTRL_16C | BGCTRL_NAME(DEFAULT_MAP)
            | BGCTRL_H32 | BGCTRL_V64;

  init_sound();

  return 0;
}


int agbtty_set_map(size_t in_nt)
{
  if(in_nt > 30 || (in_nt & 1))  /* if it's not even and in [0..30] */
  {
    errno = EDOM;
    return -1;
  }
  nt = in_nt;
  return 0;
}


static int agbtty_newline(void)
{
  curs_x = 0;
  if(curs_y >= 19)
  {
    scrollbase++;
    curs_y = 19;
  }
  else
  {
    curs_y++;
  }
  return agbtty_scrollback(0);
}


static int agbtty_putc_printable(int c)
{
  MAP[nt][(scrollbase + curs_y) & 63][curs_x++] = (c & 0xff) | text_attr;
  if(curs_x >= 30)
    return agbtty_newline();
  move_cursor();
  return 0;
}


int agbtty_putc(int c)
{
  switch(c)
  {
  case '\a':
    return agbtty_beep();
  case '\b':
    if(curs_x > 0)
      curs_x--;
    return 0;
  case '\t':
    while(curs_x & 7)
      agbtty_putc_printable(' ');
    return 0;
  case '\n':
  case '\v':
    return agbtty_newline();
  case '\f':
    return agbtty_cls();
  case '\r':
    curs_x = 0;
    return move_cursor();
  default:
    return agbtty_putc_printable(c);
  }
}


ssize_t agbtty_write(const void *in_buf, size_t size)
{
  const char *buf = in_buf;
  ssize_t sz = size;

  for(; size > 0; size--)
    agbtty_putc(*buf++);
  return sz;
}


int agbtty_puts(const char *buf)
{
  while(*buf)
    agbtty_putc(*buf++);
  return 0;
}


int agbtty_gotoxy(unsigned int x, unsigned int y)
{
  if(x >= 30 || y >= 20)
  {
    errno = EDOM;
    return -1;
  }
  curs_x = x;
  curs_y = y;
  return move_cursor();
}

unsigned int agbtty_textattr(unsigned int attr)
{
  text_attr = attr;
  return attr;
}

int agbtty_scrollback(size_t n_lines)
{
  if(n_lines > 44)
  {
    errno = EDOM;
    return -1;
  }

  BGSCROLL[0].x = 0;
  BGSCROLL[0].y = 8 * ((scrollbase - n_lines) & 63);
  agbtty_cur_scrollback = n_lines;
  return move_cursor();
}
