/*

bin2h: convert a binary file to a C array of bytes
for gfx/foo3.bin it'll write gfx_foo_bin (an array of char)
and gfx_foo_bin_len (an unsigned int)
for 4bit.chr it'll write _4bit_chr and _4bit_chr_len


Copyright 2001 Damian Yerrick

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF
OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
IN THE SOFTWARE.

*/



/**************************************\
* bin2h.c                              *
* Convert a binary file to a C header. *
*                                      ********************************\
* Copyright 2001 Damian Yerrick                                        *
\**********************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

/* strnident() *************************
   Print the closest valid C identifier to a given word.
*/
void strnident(FILE *fout, const char *src)
{
  char got_first = 0;

  while(*src != 0)
    {
      int s = *src++;

      /* initial digit  */
      if(isdigit(s) && !got_first)
	fputc('_', fout);  /* stick a '_' before an initial digit */

      /* convert only out-of-range characters */
      if(!isalpha(s) && !isdigit(s) && (s != '_'))
	{
	  if(s == '-' || s == '.' || s == '/')
	    s = '_';
	  else
	    s = 0;
	}

      if(s)
	{
	  fputc(s, fout);
	  got_first = 1;
	}
    }
}


int main(int argc, char **argv)
{
  FILE *fin;
  size_t filelen;
  int linelen;
  int arg;

  if(argc < 2)
    {
      fputs("typical usage: bin2h foo.bin bar.bin baz.bin > foo.h\n", stderr);
      return 1;
    }

  for(arg = 1; arg < argc; arg++)
    {
      /* fprintf(stderr, "arg %d: %s\n", arg, argv[arg]); */
      fin = fopen(argv[arg], "rb");
      if(!fin)
	{
	  fputs("bin2h could not open ", stderr);
          perror(argv[arg]);
	  return 1;
	}

      fseek(fin, 0, SEEK_END);
      filelen = ftell(fin);
      rewind(fin);

      if(filelen == 0)
	{
	  fclose(fin);
          fprintf(stderr, "bin2h: Warning: skipping empty file %s\n",
                  argv[arg]);
          continue;
	}


      /* two-fold: 1. provide length, and 2. align to 32-bit boundary */
      fputs("const unsigned int ", stdout);
      strnident(stdout, argv[arg]);
      printf("_len = %lu;\n", (unsigned long)filelen);

      fputs("/* generated by bin2h */\n"
	    "const unsigned char ", stdout);
      strnident(stdout, argv[arg]);
      printf("[%lu]={\n", (unsigned long)filelen);

      linelen = 0;

      while(filelen > 0)
	{
	  unsigned char c = fgetc(fin);

	  printf("%3u", (unsigned int)c);
	  filelen--;

	  /* don't put a comma after the last item */
	  if(filelen)
	    {
	      putchar(',');

	      /* break after every 16th number */
	      if(++linelen >= 16)
		{
		  linelen = 0;
		  putchar('\n');
		}
	    }
	}
      fputs("\n};\n", stdout);

      fclose(fin);
    }
  return 0;
}

