/*
 * Decompiled with CFR 0.152.
 */
package com.cypress.clocksolver;

import com.cypress.clocksolver.ExitCode;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Locale;
import java.util.Map;

public class FllSolver {
    public static final String ID = "FLL";
    private static final int ARG_MODE_IDX = 1;
    private static final int ARG_AUTO_IDX_SRC_FREQ = 2;
    private static final int ARG_AUTO_IDX_TARGET_FREQ = 3;
    private static final int ARG_AUTO_IDX_WCO_SRC = 4;
    private static final int ARG_AUTO_IDX_SRC_ACCURACY = 5;
    private static final int ARG_MANUAL_IDX_SRC_FREQ = 2;
    private static final int ARG_MANUAL_IDX_MULT = 3;
    private static final int ARG_MANUAL_IDX_DIV = 4;
    private static final int ARG_MANUAL_IDX_LOCK_TOLERANCE = 5;
    private static final int ARG_MANUAL_IDX_WCO_SRC = 6;
    private static final int ARG_MANUAL_IDX_SRC_ACCURACY = 7;
    private static final int MIN_CCO_OUTPUT_FREQ = 48;
    private static final int MIN_OUTPUT_FREQ = 24;
    private static final int MAX_OUTPUT_FREQ = 100;
    private static final double SRC_CCO_FREQ_RATIO = 2.2;
    private static final int FLOATING_POINT_FRATIONAL_DIGITS = 4;
    private static boolean OUTPUT_DIV_ENABLED = true;
    private static final String[] CCO_RANGE = new String[]{"CY_SYSCLK_FLL_CCO_RANGE0", "CY_SYSCLK_FLL_CCO_RANGE1", "CY_SYSCLK_FLL_CCO_RANGE2", "CY_SYSCLK_FLL_CCO_RANGE3", "CY_SYSCLK_FLL_CCO_RANGE4"};
    private static final double[] TRIM_STEPS = new double[]{0.0011034f, 0.001102f, 0.0011f, 0.0011f, 0.00117062f};
    private static final double[] F_MARGIN = new double[]{4.36E7, 5.81E7, 7.72E7, 1.03E8, 1.32E8};
    private static final String CONFIG_OUTPUT_MODE = "CY_SYSCLK_FLLPLL_OUTPUT_OUTPUT";

    public static ExitCode solve(String[] args, Map<String, String> computedValues) {
        ExitCode retVal;
        if (args.length < 2) {
            FllSolver.printArgCountError();
            retVal = ExitCode.ERROR_ARG_COUNT;
        } else {
            try {
                Mode mode = Mode.parse(args[1]);
                switch (mode) {
                    case AUTO: {
                        retVal = FllSolver.parseAndRunAuto(args, computedValues);
                        break;
                    }
                    case MANUAL: {
                        retVal = FllSolver.parseAndRunManual(args, computedValues);
                        break;
                    }
                    default: {
                        retVal = ExitCode.ERROR_ARG_VALUE;
                        break;
                    }
                }
            }
            catch (NumberFormatException e) {
                System.err.println("Unable to parse argument values: " + e.getMessage());
                retVal = ExitCode.ERROR_ARG_VALUE;
            }
        }
        return retVal;
    }

    private static ExitCode parseAndRunAuto(String[] args, Map<String, String> computedValues) {
        ExitCode retVal;
        if (args.length == 6) {
            double srcFreqMHz = Double.parseDouble(args[2]);
            double targetFreqMHz = Double.parseDouble(args[3]);
            boolean wcoIsSource = Boolean.parseBoolean(args[4]);
            double srcAccuracyPercent = Double.parseDouble(args[5]);
            retVal = FllSolver.verifyDesiredFrequency(srcFreqMHz, targetFreqMHz, OUTPUT_DIV_ENABLED);
            if (retVal == ExitCode.SUCCESS) {
                double srcAccuracy = Math.abs(srcAccuracyPercent / 100.0);
                retVal = FllSolver.solveAuto(srcFreqMHz, targetFreqMHz, wcoIsSource, srcAccuracy, computedValues);
            }
        } else {
            FllSolver.printArgCountError();
            retVal = ExitCode.ERROR_ARG_COUNT;
        }
        return retVal;
    }

    private static ExitCode parseAndRunManual(String[] args, Map<String, String> computedValues) {
        ExitCode retVal;
        if (args.length == 8) {
            double srcFreqMHz = Double.parseDouble(args[2]);
            int mult = Integer.parseInt(args[3]);
            int div = Integer.parseInt(args[4]);
            int lockTol = Integer.parseInt(args[5]);
            boolean wcoIsSource = Boolean.parseBoolean(args[6]);
            double srcAccuracyPercent = Double.parseDouble(args[7]);
            double expectedFreq = srcFreqMHz * (double)mult / (double)div / (double)(OUTPUT_DIV_ENABLED ? 2 : 1);
            retVal = FllSolver.verifyDesiredFrequency(srcFreqMHz, expectedFreq, OUTPUT_DIV_ENABLED);
            if (retVal == ExitCode.SUCCESS) {
                double srcAccuracy = Math.abs(srcAccuracyPercent / 100.0);
                retVal = FllSolver.solveManual(srcFreqMHz, mult, div, lockTol, wcoIsSource, srcAccuracy, computedValues);
            }
        } else {
            FllSolver.printArgCountError();
            retVal = ExitCode.ERROR_ARG_COUNT;
        }
        return retVal;
    }

    public static ExitCode solveAuto(double srcFreqMHz, double targetFreqMHz, boolean wcoIsSource, double srcAccuracy, Map<String, String> computedValues) {
        int configPgain;
        int configIgain;
        boolean configEnableOutputDiv = OUTPUT_DIV_ENABLED;
        double ccoFreqMHz = targetFreqMHz * (double)(configEnableOutputDiv ? 2 : 1);
        int ccoIdx = FllSolver.getCcoRangeIdx(ccoFreqMHz);
        String configCcoRange = CCO_RANGE[ccoIdx];
        int configRefDiv = wcoIsSource ? 19 : (int)Math.ceil(srcFreqMHz / targetFreqMHz * 250.0);
        int configFllMult = (int)Math.ceil(ccoFreqMHz / (srcFreqMHz / (double)configRefDiv));
        double CCO_ACCURACY = 0.0025;
        int configLockTolerance = (int)Math.ceil((double)configFllMult * (1.5 * (1.0025 / (1.0 - srcAccuracy) - 1.0)));
        double kcco = TRIM_STEPS[ccoIdx] * F_MARGIN[ccoIdx] / 1000.0;
        double ki_p = 850.0 / (kcco * ((double)configRefDiv / srcFreqMHz));
        double[] gains = new double[]{0.00390625, 0.0078125, 0.015625, 0.03125, 0.0625, 0.125, 0.25, 0.5, 1.0, 2.0, 4.0, 8.0};
        for (configIgain = gains.length - 1; gains[configIgain] > ki_p && configIgain != 0; --configIgain) {
        }
        if (wcoIsSource && configIgain > 0) {
            --configIgain;
        }
        for (configPgain = gains.length - 1; gains[configPgain] > ki_p - gains[configIgain] && configPgain != 0; --configPgain) {
        }
        if (wcoIsSource && configPgain > 0) {
            --configPgain;
        }
        int configCcoFreq = (int)Math.floor(Math.log(ccoFreqMHz * 1000000.0 / F_MARGIN[ccoIdx]) / Math.log(1.0 + TRIM_STEPS[ccoIdx]));
        int configSettlingCount = FllSolver.getSettlingCount(wcoIsSource, srcFreqMHz, ccoFreqMHz, configRefDiv);
        double accuracy = FllSolver.calculateFLLAccuracy(ccoFreqMHz, srcFreqMHz, TRIM_STEPS[ccoIdx], configRefDiv, configLockTolerance);
        double accuracyPercent = 100.0 * accuracy;
        String accuracyString = FllSolver.formatDoubleInsensitive(accuracyPercent);
        computedValues.put("fllMult", Integer.toString(configFllMult));
        computedValues.put("refDiv", Integer.toString(configRefDiv));
        computedValues.put("ccoRange", configCcoRange);
        computedValues.put("enableOutputDiv", Boolean.toString(configEnableOutputDiv));
        computedValues.put("lockTolerance", Integer.toString(configLockTolerance));
        computedValues.put("igain", Integer.toString(configIgain));
        computedValues.put("pgain", Integer.toString(configPgain));
        computedValues.put("settlingCount", Integer.toString(configSettlingCount));
        computedValues.put("outputMode", CONFIG_OUTPUT_MODE);
        computedValues.put("cco_Freq", Integer.toString(configCcoFreq));
        computedValues.put("accuracy", accuracyString);
        return ExitCode.SUCCESS;
    }

    public static ExitCode solveManual(double srcFreqMHz, int configFllMult, int configRefDiv, int configLockTolerance, boolean wcoIsSource, double srcAccuracy, Map<String, String> computedValues) {
        int configPgain;
        int configIgain;
        boolean configEnableOutputDiv = OUTPUT_DIV_ENABLED;
        double outputDiv = configEnableOutputDiv ? 2.0 : 1.0;
        double actualFreq = srcFreqMHz * (double)configFllMult / (double)configRefDiv / outputDiv;
        double ccoFreqMHz = actualFreq * outputDiv;
        int ccoIdx = FllSolver.getCcoRangeIdx(ccoFreqMHz);
        String configCcoRange = CCO_RANGE[ccoIdx];
        double kcco = TRIM_STEPS[ccoIdx] * F_MARGIN[ccoIdx] / 1000.0;
        double ki_p = 850.0 / (kcco * ((double)configRefDiv / srcFreqMHz));
        double[] gains = new double[]{0.00390625, 0.0078125, 0.015625, 0.03125, 0.0625, 0.125, 0.25, 0.5, 1.0, 2.0, 4.0, 8.0};
        for (configIgain = gains.length - 1; gains[configIgain] > ki_p && configIgain != 0; --configIgain) {
        }
        if (wcoIsSource && configIgain > 0) {
            --configIgain;
        }
        for (configPgain = gains.length - 1; gains[configPgain] > ki_p - gains[configIgain] && configPgain != 0; --configPgain) {
        }
        if (wcoIsSource && configPgain > 0) {
            --configPgain;
        }
        int configCcoFreq = (int)Math.floor(Math.log(ccoFreqMHz * 1000000.0 / F_MARGIN[ccoIdx]) / Math.log(1.0 + TRIM_STEPS[ccoIdx]));
        int configSettlingCount = FllSolver.getSettlingCount(wcoIsSource, srcFreqMHz, ccoFreqMHz, configRefDiv);
        double accuracy = FllSolver.calculateFLLAccuracy(ccoFreqMHz, srcFreqMHz, TRIM_STEPS[ccoIdx], configRefDiv, configLockTolerance);
        double accuracyPercent = 100.0 * accuracy;
        String accuracyString = FllSolver.formatDoubleInsensitive(accuracyPercent);
        computedValues.put("fllMult", Integer.toString(configFllMult));
        computedValues.put("refDiv", Integer.toString(configRefDiv));
        computedValues.put("ccoRange", configCcoRange);
        computedValues.put("enableOutputDiv", Boolean.toString(configEnableOutputDiv));
        computedValues.put("lockTolerance", Integer.toString(configLockTolerance));
        computedValues.put("igain", Integer.toString(configIgain));
        computedValues.put("pgain", Integer.toString(configPgain));
        computedValues.put("settlingCount", Integer.toString(configSettlingCount));
        computedValues.put("outputMode", CONFIG_OUTPUT_MODE);
        computedValues.put("cco_Freq", Integer.toString(configCcoFreq));
        computedValues.put("accuracy", accuracyString);
        return ExitCode.SUCCESS;
    }

    private static int getSettlingCount(boolean isWco, double srcFreqMHz, double ccoFreqMHz, int configRefDiv) {
        double ttref = (double)configRefDiv / (srcFreqMHz * 1000.0);
        double testval = 6.0 / (ccoFreqMHz * 1000.0);
        double divval = Math.ceil(srcFreqMHz);
        double altval = Math.ceil(divval / ttref + 1.0);
        return (int)(isWco ? 200.0 : (ttref > testval ? divval : (divval > altval ? divval : altval)));
    }

    private static int getCcoRangeIdx(double ccoFreqMHz) {
        int range = ccoFreqMHz >= 150.3392 ? 4 : (ccoFreqMHz >= 113.00938 ? 3 : (ccoFreqMHz >= 84.9487 ? 2 : (ccoFreqMHz >= 63.8556 ? 1 : 0)));
        return range;
    }

    private static double calculateFLLAccuracy(double targetCcoFreq, double sourceRefFreq, double trimStep, int refDiv, int lockTolerance) {
        double measure = 1.0 / (double)refDiv * (sourceRefFreq / targetCcoFreq);
        double ss_2 = trimStep / 2.0;
        double fflPrecision = Math.max(measure, ss_2);
        return fflPrecision * (double)(lockTolerance + 2);
    }

    private static ExitCode verifyDesiredFrequency(double sourceFreq, double targetFreq, boolean outputDiv) {
        ExitCode retVal;
        double ratio;
        double d = ratio = outputDiv ? 1.1 : 2.2;
        if (targetFreq < 24.0 || 100.0 < targetFreq) {
            System.err.println(String.format("Invalid FLL target frequency '%1$f'. Must be within the rage %2$d-%3$d.", targetFreq, 24, 100));
            retVal = ExitCode.ERROR_ARG_VALUE;
        } else if (sourceFreq * ratio >= targetFreq) {
            System.err.println(String.format("Invalid FLL source/target frequency ratio. Target must be at least %1$fx faster than the source.", ratio));
            retVal = ExitCode.ERROR_ARG_VALUE;
        } else {
            retVal = ExitCode.SUCCESS;
        }
        return retVal;
    }

    private static String formatDoubleInsensitive(double value) {
        NumberFormat formatter = DecimalFormat.getInstance(Locale.ROOT);
        formatter.setMinimumFractionDigits(4);
        return formatter.format(value);
    }

    private static void printArgCountError() {
        System.err.println("FLL Solver requires 5 arguments in automatic mode:");
        System.err.println("\t\"auto\" automatic - Use automatic solver based on desired frequency");
        System.err.println("\tdouble srcFreqMHz - Source clock frequency for the FLL in MHz.");
        System.err.println("\tdouble targetFreqMHz - Output frequency of the FLL in MHz.");
        System.err.println("\tboolean wcoIsSource - Is the WCO sourcing the FLL?");
        System.err.println("\tdouble sourceAccuracy - FLL reference clock accuracy (%)?");
        System.err.println("FLL Solver requires 7 arguments in manual mode:");
        System.err.println("\t\"manual\" automatic - Use automatic solver based on desired frequency");
        System.err.println("\tdouble srcFreqMHz - Source clock frequency for the FLL in MHz.");
        System.err.println("\tint Mult - Clock frequency multiplier.");
        System.err.println("\tint Div - Clock frequency divider.");
        System.err.println("\tint lockTolerance - FLL lock tolerance.");
        System.err.println("\tboolean wcoIsSource - Is the WCO sourcing the FLL?");
        System.err.println("\tdouble sourceAccuracy - FLL reference clock accuracy (%)?");
    }

    private static enum Mode {
        AUTO,
        MANUAL,
        UNDEFINED;


        public static Mode parse(String value) {
            switch (value) {
                case "auto": {
                    return AUTO;
                }
                case "manual": {
                    return MANUAL;
                }
            }
            return UNDEFINED;
        }
    }
}

