/***************************************************************************//**
* \file SDIO_HOST_cfg.c
*
* \brief
*  This file provides the configuration of the UDB based SDIO driver.
*
********************************************************************************
* \copyright
* Copyright 2016-2020 Cypress Semiconductor Corporation
* SPDX-License-Identifier: Apache-2.0
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*******************************************************************************/

#include "SDIO_HOST_cfg.h"

#if defined(__cplusplus)
extern "C" {
#endif

/***************************CMD DMA Config Struct****************************/
cy_stc_dma_descriptor_config_t SDIO_HOST_CMD_DMA_CMD_DMA_Desc_config =
{
    .retrigger       = CY_DMA_RETRIG_16CYC,
    .interruptType   = CY_DMA_1ELEMENT,
    .triggerOutType  = CY_DMA_1ELEMENT,
    .channelState    = CY_DMA_CHANNEL_DISABLED,
    .triggerInType   = CY_DMA_1ELEMENT,
    .dataSize        = CY_DMA_BYTE,
    .srcTransferSize = CY_DMA_TRANSFER_SIZE_DATA,
    .dstTransferSize = CY_DMA_TRANSFER_SIZE_DATA,
    .descriptorType  = CY_DMA_1D_TRANSFER,
    .srcAddress      = NULL,
    .dstAddress      = NULL,
    .srcXincrement   = 1L,
    .dstXincrement   = 0L,
    .xCount          = 5UL,
    .srcYincrement   = 0L,
    .dstYincrement   = 0L,
    .yCount          = 1UL,
    .nextDescriptor  = NULL
};

cy_stc_dma_descriptor_t SDIO_HOST_CMD_DMA_CMD_DMA_Desc =
{
    .ctl = 0UL,
    .src = 0UL,
    .dst = 0UL,
    .xCtl = 0UL,
    .yCtl = 0UL,
    .nextPtr = 0UL
};

/***************************Read DMA Config Struct****************************/
cy_stc_dma_descriptor_config_t SDIO_HOST_Read_DMA_Read_DMA_Desc_config =
{
    .retrigger       = CY_DMA_RETRIG_IM,
    .interruptType   = CY_DMA_DESCR,
    .triggerOutType  = CY_DMA_1ELEMENT,
    .channelState    = CY_DMA_CHANNEL_DISABLED,
    .triggerInType   = CY_DMA_X_LOOP,
    .dataSize        = CY_DMA_HALFWORD,
    .srcTransferSize = CY_DMA_TRANSFER_SIZE_DATA,
    .dstTransferSize = CY_DMA_TRANSFER_SIZE_DATA,
    .descriptorType  = CY_DMA_2D_TRANSFER,
    .srcAddress      = NULL,
    .dstAddress      = NULL,
    .srcXincrement   = 0L,
    .dstXincrement   = 2L,
    .xCount          = 10UL,
    .srcYincrement   = 0L,
    .dstYincrement   = 10L,
    .yCount          = 2UL,
    .nextDescriptor  = NULL
};

cy_stc_dma_descriptor_t SDIO_HOST_Read_DMA_Read_DMA_Desc =
{
    .ctl = 0UL,
    .src = 0UL,
    .dst = 0UL,
    .xCtl = 0UL,
    .yCtl = 0UL,
    .nextPtr = 0UL
};

/***************************Resp DMA Config Struct****************************/
cy_stc_dma_descriptor_config_t SDIO_HOST_Resp_DMA_Resp_DMA_Desc_config =
{
    .retrigger       = CY_DMA_RETRIG_IM,
    .interruptType   = CY_DMA_1ELEMENT,
    .triggerOutType  = CY_DMA_1ELEMENT,
    .channelState    = CY_DMA_CHANNEL_DISABLED,
    .triggerInType   = CY_DMA_1ELEMENT,
    .dataSize        = CY_DMA_BYTE,
    .srcTransferSize = CY_DMA_TRANSFER_SIZE_DATA,
    .dstTransferSize = CY_DMA_TRANSFER_SIZE_DATA,
    .descriptorType  = CY_DMA_1D_TRANSFER,
    .srcAddress      = NULL,
    .dstAddress      = NULL,
    .srcXincrement   = 0L,
    .dstXincrement   = 1L,
    .xCount          = 6UL,
    .srcYincrement   = 0L,
    .dstYincrement   = 0L,
    .yCount          = 1UL,
    .nextDescriptor  = NULL
};

cy_stc_dma_descriptor_t SDIO_HOST_Resp_DMA_Resp_DMA_Desc =
{
    .ctl = 0UL,
    .src = 0UL,
    .dst = 0UL,
    .xCtl = 0UL,
    .yCtl = 0UL,
    .nextPtr = 0UL
};

/***************************Write DMA Config Struct****************************/
cy_stc_dma_descriptor_config_t SDIO_HOST_Write_DMA_Write_DMA_Desc_config =
{
    .retrigger       = CY_DMA_RETRIG_4CYC,
    .interruptType   = CY_DMA_DESCR,
    .triggerOutType  = CY_DMA_1ELEMENT,
    .channelState    = CY_DMA_CHANNEL_DISABLED,
    .triggerInType   = CY_DMA_X_LOOP,
    .dataSize        = CY_DMA_HALFWORD,
    .srcTransferSize = CY_DMA_TRANSFER_SIZE_DATA,
    .dstTransferSize = CY_DMA_TRANSFER_SIZE_DATA,
    .descriptorType  = CY_DMA_2D_TRANSFER,
    .srcAddress      = NULL,
    .dstAddress      = NULL,
    .srcXincrement   = 2L,
    .dstXincrement   = 0L,
    .xCount          = 10UL,
    .srcYincrement   = 10L,
    .dstYincrement   = 0L,
    .yCount          = 2UL,
    .nextDescriptor  = NULL
};

cy_stc_dma_descriptor_t SDIO_HOST_Write_DMA_Write_DMA_Desc =
{
    .ctl = 0UL,
    .src = 0UL,
    .dst = 0UL,
    .xCtl = 0UL,
    .yCtl = 0UL,
    .nextPtr = 0UL
};



/***************UDB Config code *****************/

#define CY_CFG_BASE_ADDR_COUNT 12u

#if defined(__GNUC__) || defined(__ARMCC_VERSION)
    #define CYPACKED
    #define CYPACKED_ATTR __attribute__ ((packed))
    #define CY_CFG_UNUSED __attribute__ ((unused))


#elif defined(__ICCARM__)
    #include <intrinsics.h>

    #define CYPACKED __packed
    #define CYPACKED_ATTR
    #define CY_CFG_UNUSED _Pragma("diag_suppress=Pe177")


#else
    #error Unsupported toolchain
#endif


#ifndef CYCODE
    #define CYCODE
#endif
#ifndef CYFAR
    #define CYFAR
#endif


CY_CFG_UNUSED
static void CYMEMZERO(void *s, size_t n);
CY_CFG_UNUSED
static void CYMEMZERO(void *s, size_t n)
{
    (void)memset(s, 0, n);
}
CY_CFG_UNUSED
static void CYCONFIGCPY(void *dest, const void *src, size_t n);
CY_CFG_UNUSED
static void CYCONFIGCPY(void *dest, const void *src, size_t n)
{
    (void)memcpy(dest, src, n);
}
CY_CFG_UNUSED
static void CYCONFIGCPYCODE(void *dest, const void *src, size_t n);
CY_CFG_UNUSED
static void CYCONFIGCPYCODE(void *dest, const void *src, size_t n)
{
    (void)memcpy(dest, src, n);
}

CYPACKED typedef struct
{
    uint8 offset;
    uint8 value;
} CYPACKED_ATTR cy_cfg_addrvalue_t;


/*******************************************************************************
* Function Name: cfg_write_bytes32
********************************************************************************
* Summary:
*  This function is used for setting up the chip configuration areas that
*  contain relatively sparse data.
*
* Parameters:
*   void
*
* Return:
*   void
*
*******************************************************************************/

static void cfg_write_bytes32(const uint32 addr_table[], const cy_cfg_addrvalue_t data_table[]);
static void cfg_write_bytes32(const uint32 addr_table[], const cy_cfg_addrvalue_t data_table[])
{
    /* For 32-bit little-endian architectures */
    uint32 i, j = 0u;
    for (i = 0u; i < CY_CFG_BASE_ADDR_COUNT; i++)
    {
        uint32 baseAddr = addr_table[i];
        uint8 count = (uint8)baseAddr;
        baseAddr &= 0xFFFFFF00u;
        while (count != 0u)
        {
            CY_SET_REG8((void *)(baseAddr + data_table[j].offset), data_table[j].value);
            j++;
            count--;
        }
    }
}

static const uint32 CYCODE cy_cfg_addr_table[] = 
{
    0x40340002u, /* Base address: 0x40340000 Count: 2 */
    0x4034010Au, /* Base address: 0x40340100 Count: 10 */
    0x40340301u, /* Base address: 0x40340300 Count: 1 */
    0x40340405u, /* Base address: 0x40340400 Count: 5 */
    0x4034205Cu, /* Base address: 0x40342000 Count: 92 */
    0x40342238u, /* Base address: 0x40342200 Count: 56 */
    0x4034242Cu, /* Base address: 0x40342400 Count: 44 */
    0x4034262Eu, /* Base address: 0x40342600 Count: 46 */
    0x40342837u, /* Base address: 0x40342800 Count: 55 */
    0x40342A29u, /* Base address: 0x40342A00 Count: 41 */
    0x40347104u, /* Base address: 0x40347100 Count: 4 */
    0x40347804u, /* Base address: 0x40347800 Count: 4 */
};

static const cy_cfg_addrvalue_t CYCODE cy_cfg_data_table[] = 
{
    {0x00u, 0xFFu},
    {0x09u, 0x01u},
    {0x09u, 0x01u},
    {0x0Du, 0x02u},
    {0x10u, 0x10u},
    {0x14u, 0x88u},
    {0x18u, 0x10u},
    {0x1Cu, 0x88u},
    {0x20u, 0x10u},
    {0x24u, 0x88u},
    {0x28u, 0x10u},
    {0x2Cu, 0x88u},
    {0x19u, 0x40u},
    {0x10u, 0x07u},
    {0x18u, 0x47u},
    {0x24u, 0x03u},
    {0x28u, 0x0Fu},
    {0x2Cu, 0x2Fu},
    {0x00u, 0x01u},
    {0x0Eu, 0x0Eu},
    {0x10u, 0x08u},
    {0x18u, 0x01u},
    {0x1Au, 0x08u},
    {0x1Cu, 0x01u},
    {0x1Eu, 0x02u},
    {0x22u, 0x01u},
    {0x28u, 0x01u},
    {0x2Au, 0x04u},
    {0x2Eu, 0x01u},
    {0x30u, 0x0Fu},
    {0x3Cu, 0x02u},
    {0x40u, 0x63u},
    {0x41u, 0x04u},
    {0x42u, 0x10u},
    {0x44u, 0x02u},
    {0x45u, 0x0Fu},
    {0x46u, 0xECu},
    {0x47u, 0xB0u},
    {0x48u, 0x2Du},
    {0x49u, 0xFFu},
    {0x4Au, 0xFFu},
    {0x4Bu, 0xFFu},
    {0x4Cu, 0x02u},
    {0x4Fu, 0x0Cu},
    {0x50u, 0x08u},
    {0x51u, 0x10u},
    {0x5Cu, 0x0Cu},
    {0x5Eu, 0x0Cu},
    {0x63u, 0x09u},
    {0x65u, 0x08u},
    {0x68u, 0xC0u},
    {0x6Cu, 0x10u},
    {0x6Du, 0x11u},
    {0x6Eu, 0x40u},
    {0x6Fu, 0x01u},
    {0x71u, 0x10u},
    {0x72u, 0x50u},
    {0x73u, 0xA8u},
    {0x81u, 0x01u},
    {0x83u, 0x02u},
    {0x85u, 0x02u},
    {0x89u, 0x02u},
    {0x8Du, 0x02u},
    {0x91u, 0x30u},
    {0x93u, 0x04u},
    {0x95u, 0x06u},
    {0x97u, 0x38u},
    {0x99u, 0x20u},
    {0x9Au, 0x01u},
    {0x9Bu, 0x10u},
    {0x9Du, 0x06u},
    {0x9Fu, 0x38u},
    {0xA3u, 0x02u},
    {0xA5u, 0x02u},
    {0xA6u, 0x01u},
    {0xABu, 0x01u},
    {0xB1u, 0x01u},
    {0xB2u, 0x01u},
    {0xB3u, 0x20u},
    {0xB5u, 0x10u},
    {0xB7u, 0x0Eu},
    {0xBDu, 0x80u},
    {0xBEu, 0x04u},
    {0xBFu, 0x14u},
    {0xC0u, 0x54u},
    {0xC1u, 0x06u},
    {0xC2u, 0x32u},
    {0xC6u, 0xF0u},
    {0xC7u, 0x0Eu},
    {0xC8u, 0x18u},
    {0xC9u, 0xFFu},
    {0xCAu, 0xFFu},
    {0xCBu, 0xFFu},
    {0xCFu, 0x2Fu},
    {0xD0u, 0x08u},
    {0xD1u, 0x10u},
    {0xDCu, 0x04u},
    {0xDDu, 0x0Cu},
    {0xDEu, 0x0Cu},
    {0xDFu, 0x04u},
    {0xE3u, 0x09u},
    {0xE4u, 0x50u},
    {0xE5u, 0xA8u},
    {0xE6u, 0x08u},
    {0xE7u, 0x03u},
    {0xE8u, 0x08u},
    {0xEAu, 0x18u},
    {0xEBu, 0x03u},
    {0xF0u, 0x58u},
    {0xF1u, 0xECu},
    {0x00u, 0x18u},
    {0x01u, 0x15u},
    {0x03u, 0x20u},
    {0x04u, 0x43u},
    {0x06u, 0x34u},
    {0x07u, 0x35u},
    {0x08u, 0x40u},
    {0x0Cu, 0x5Au},
    {0x0Eu, 0x25u},
    {0x0Fu, 0x4Au},
    {0x11u, 0x20u},
    {0x13u, 0x15u},
    {0x14u, 0x40u},
    {0x15u, 0x40u},
    {0x17u, 0x0Au},
    {0x18u, 0x20u},
    {0x19u, 0x10u},
    {0x1Bu, 0x25u},
    {0x1Cu, 0x40u},
    {0x1Du, 0x0Au},
    {0x1Fu, 0x40u},
    {0x20u, 0x20u},
    {0x21u, 0x0Cu},
    {0x23u, 0x03u},
    {0x24u, 0x40u},
    {0x2Au, 0x20u},
    {0x2Cu, 0x45u},
    {0x2Eu, 0x3Au},
    {0x30u, 0x40u},
    {0x32u, 0x0Eu},
    {0x33u, 0x70u},
    {0x34u, 0x01u},
    {0x35u, 0x0Cu},
    {0x36u, 0x31u},
    {0x37u, 0x03u},
    {0x3Au, 0xA8u},
    {0x3Cu, 0xA8u},
    {0x3Eu, 0x01u},
    {0x3Fu, 0x54u},
    {0x40u, 0x42u},
    {0x41u, 0x05u},
    {0x45u, 0x20u},
    {0x46u, 0x06u},
    {0x48u, 0x06u},
    {0x49u, 0xFFu},
    {0x4Au, 0xFFu},
    {0x4Bu, 0xFFu},
    {0x4Du, 0xA0u},
    {0x5Cu, 0x0Cu},
    {0x5Du, 0x04u},
    {0x5Eu, 0x0Cu},
    {0x63u, 0x09u},
    {0x66u, 0x40u},
    {0x67u, 0x40u},
    {0x68u, 0x10u},
    {0x69u, 0x50u},
    {0x02u, 0x60u},
    {0x08u, 0x87u},
    {0x0Au, 0x78u},
    {0x0Eu, 0x08u},
    {0x12u, 0x21u},
    {0x14u, 0x04u},
    {0x18u, 0xB9u},
    {0x1Au, 0x06u},
    {0x1Cu, 0x89u},
    {0x1Eu, 0x72u},
    {0x20u, 0x04u},
    {0x24u, 0x8Eu},
    {0x26u, 0x51u},
    {0x2Au, 0x04u},
    {0x2Eu, 0x01u},
    {0x30u, 0x80u},
    {0x34u, 0x7Fu},
    {0x3Au, 0x20u},
    {0x3Cu, 0x20u},
    {0x3Eu, 0x01u},
    {0x40u, 0x43u},
    {0x41u, 0x02u},
    {0x44u, 0x01u},
    {0x49u, 0xFFu},
    {0x4Au, 0xFFu},
    {0x4Bu, 0xFFu},
    {0x4Cu, 0x12u},
    {0x4Du, 0x5Cu},
    {0x4Eu, 0x78u},
    {0x57u, 0x02u},
    {0x58u, 0x77u},
    {0x5Cu, 0x0Cu},
    {0x5Eu, 0x0Cu},
    {0x5Fu, 0x0Cu},
    {0x62u, 0x08u},
    {0x63u, 0x09u},
    {0x64u, 0x50u},
    {0x65u, 0xA8u},
    {0x69u, 0x1Cu},
    {0x6Au, 0x58u},
    {0x6Bu, 0xA1u},
    {0x6Du, 0x10u},
    {0x70u, 0x10u},
    {0x71u, 0x1Du},
    {0x80u, 0x40u},
    {0x84u, 0x8Eu},
    {0x86u, 0x71u},
    {0x88u, 0x40u},
    {0x8Eu, 0x21u},
    {0x90u, 0x40u},
    {0x92u, 0x02u},
    {0x96u, 0x18u},
    {0x98u, 0x03u},
    {0x9Au, 0x04u},
    {0x9Cu, 0xCCu},
    {0x9Eu, 0x33u},
    {0xA0u, 0x08u},
    {0xA4u, 0x40u},
    {0xA8u, 0x46u},
    {0xAAu, 0xB1u},
    {0xACu, 0x10u},
    {0xAEu, 0x08u},
    {0xB2u, 0xF8u},
    {0xB4u, 0x07u},
    {0xB6u, 0x07u},
    {0xBAu, 0xA8u},
    {0xBCu, 0xA8u},
    {0xC0u, 0x43u},
    {0xC1u, 0x02u},
    {0xC4u, 0x01u},
    {0xC5u, 0x40u},
    {0xC6u, 0xB0u},
    {0xC8u, 0x0Au},
    {0xC9u, 0xFFu},
    {0xCAu, 0xFFu},
    {0xCBu, 0xFFu},
    {0xCCu, 0x13u},
    {0xCDu, 0x5Cu},
    {0xCEu, 0x77u},
    {0xDCu, 0x0Cu},
    {0xDEu, 0x0Cu},
    {0xE3u, 0x09u},
    {0xE4u, 0x50u},
    {0xE5u, 0xA8u},
    {0xE9u, 0x1Cu},
    {0xEAu, 0x58u},
    {0xEBu, 0xA1u},
    {0xEDu, 0x10u},
    {0xF0u, 0x10u},
    {0xF1u, 0x1Du},
    {0x00u, 0x40u},
    {0x01u, 0x14u},
    {0x03u, 0x20u},
    {0x04u, 0x33u},
    {0x05u, 0x3Cu},
    {0x06u, 0x44u},
    {0x08u, 0x97u},
    {0x09u, 0x3Cu},
    {0x0Au, 0x48u},
    {0x0Du, 0x01u},
    {0x11u, 0x03u},
    {0x12u, 0x91u},
    {0x15u, 0x03u},
    {0x16u, 0x08u},
    {0x1Bu, 0x03u},
    {0x1Cu, 0x13u},
    {0x1Eu, 0xECu},
    {0x1Fu, 0x03u},
    {0x20u, 0xADu},
    {0x21u, 0x04u},
    {0x22u, 0x52u},
    {0x27u, 0x2Cu},
    {0x29u, 0x02u},
    {0x2Eu, 0x12u},
    {0x2Fu, 0x04u},
    {0x30u, 0x07u},
    {0x31u, 0x18u},
    {0x33u, 0x07u},
    {0x34u, 0xF8u},
    {0x35u, 0x20u},
    {0x39u, 0x02u},
    {0x3Au, 0x22u},
    {0x3Cu, 0x22u},
    {0x3Fu, 0x15u},
    {0x40u, 0x43u},
    {0x41u, 0x02u},
    {0x44u, 0x06u},
    {0x49u, 0xFFu},
    {0x4Au, 0xFFu},
    {0x4Bu, 0xFFu},
    {0x4Cu, 0x12u},
    {0x4Du, 0x5Cu},
    {0x4Eu, 0x78u},
    {0x5Cu, 0x0Cu},
    {0x5Du, 0x04u},
    {0x5Eu, 0x0Cu},
    {0x63u, 0x09u},
    {0x64u, 0x50u},
    {0x65u, 0xA8u},
    {0x69u, 0x1Cu},
    {0x6Au, 0x58u},
    {0x6Bu, 0xA1u},
    {0x6Du, 0x10u},
    {0x70u, 0x10u},
    {0x71u, 0x1Du},
    {0x01u, 0x20u},
    {0x0Bu, 0x40u},
    {0x0Du, 0x10u},
    {0x11u, 0x32u},
    {0x13u, 0x09u},
    {0x15u, 0x0Au},
    {0x17u, 0x31u},
    {0x23u, 0x34u},
    {0x29u, 0x33u},
    {0x2Bu, 0x08u},
    {0x2Du, 0x0Cu},
    {0x31u, 0x0Cu},
    {0x33u, 0x01u},
    {0x35u, 0x02u},
    {0x37u, 0x70u},
    {0x39u, 0x02u},
    {0x3Fu, 0x55u},
    {0x40u, 0x16u},
    {0x41u, 0x02u},
    {0x44u, 0x04u},
    {0x49u, 0xFFu},
    {0x4Au, 0xFFu},
    {0x4Bu, 0xFFu},
    {0x4Cu, 0x12u},
    {0x4Du, 0x5Cu},
    {0x4Eu, 0x78u},
    {0x57u, 0x02u},
    {0x58u, 0x76u},
    {0x5Du, 0x04u},
    {0x5Eu, 0x0Cu},
    {0x5Fu, 0x0Cu},
    {0x62u, 0x08u},
    {0x63u, 0x09u},
    {0x64u, 0x50u},
    {0x65u, 0xA8u},
    {0x69u, 0x1Cu},
    {0x6Au, 0x58u},
    {0x6Bu, 0xA1u},
    {0x6Du, 0x10u},
    {0x70u, 0x10u},
    {0x71u, 0x1Du},
    {0xE8u, 0x63u},
    {0xECu, 0x53u},
    {0xF0u, 0x54u},
    {0xF4u, 0x01u},
    {0x00u, 0x01u},
    {0x10u, 0x01u},
    {0x14u, 0x01u},
    {0x18u, 0x01u},
};



CYPACKED typedef struct 
{
    void *address;
    uint16 size;
} CYPACKED_ATTR cfg_memset_t;

CYPACKED typedef struct 
{
    void *dest;
    const void *src;
    size_t size;
} CYPACKED_ATTR cfg_memcpy_t;

static const cfg_memset_t CYCODE cfg_memset_list[] = 
{
    /* address, size */
    {(void CYFAR *)(CYDEV_UDB_UDBPAIR0_UDBSNG0_BASE), 116u},
    {(void CYFAR *)(CYDEV_UDB_UDBPAIR0_UDBSNG1_BASE), 116u},
    {(void CYFAR *)(CYDEV_UDB_UDBPAIR1_UDBSNG0_BASE), 116u},
    {(void CYFAR *)(CYDEV_UDB_UDBPAIR2_UDBSNG0_BASE), 116u},
    {(void CYFAR *)(CYDEV_UDB_UDBPAIR3_UDBSNG1_BASE), 116u},
    {(void CYFAR *)(CYDEV_UDB_UDBPAIR4_UDBSNG0_BASE), 116u},
    {(void CYFAR *)(CYDEV_UDB_UDBPAIR5_UDBSNG0_BASE), 116u},
};

/* UDB_UDBPAIR5_UDBSNG1 Address: CYDEV_UDB_UDBPAIR5_UDBSNG1_BASE Size (bytes): 116 */
static const uint8 CYCODE BS_UDB_UDBPAIR5_UDBSNG1_VAL[] = {
    0x08u, 0x00u, 0x00u, 0x00u, 0x42u, 0x00u, 0x2Du, 0x00u, 0x08u, 0x24u, 0x00u, 0x00u, 0x22u, 0x24u, 0x45u, 0x00u, 
    0x00u, 0x10u, 0x10u, 0x08u, 0x00u, 0x70u, 0x00u, 0x83u, 0x08u, 0x24u, 0x00u, 0x00u, 0x00u, 0x4Eu, 0x00u, 0xB0u, 
    0x00u, 0x01u, 0x00u, 0x00u, 0x00u, 0x70u, 0x00u, 0x8Fu, 0x3Cu, 0x00u, 0x40u, 0x00u, 0x00u, 0x00u, 0x01u, 0x00u, 
    0x00u, 0xC0u, 0x00u, 0x1Fu, 0x60u, 0x00u, 0x1Fu, 0x20u, 0x80u, 0x00u, 0x00u, 0x00u, 0x80u, 0x08u, 0x10u, 0x41u, 
    0x46u, 0x02u, 0x00u, 0x00u, 0x05u, 0x00u, 0x04u, 0x0Bu, 0x14u, 0xFFu, 0xFFu, 0xFFu, 0x13u, 0x5Cu, 0x77u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x02u, 0x74u, 0x00u, 0x00u, 0x00u, 0x0Cu, 0x0Cu, 0x0Cu, 0x0Cu, 
    0x00u, 0x00u, 0x08u, 0x09u, 0x50u, 0xA8u, 0x00u, 0x00u, 0x00u, 0x1Cu, 0x58u, 0xA1u, 0x00u, 0x10u, 0x00u, 0x00u, 
    0x10u, 0x1Du, 0x00u, 0x00u};

/* UDB_UDBPAIR1_UDBSNG1 Address: CYDEV_UDB_UDBPAIR1_UDBSNG1_BASE Size (bytes): 116 */
static const uint8 CYCODE BS_UDB_UDBPAIR1_UDBSNG1_VAL[] = {
    0x08u, 0x0Au, 0x00u, 0xD0u, 0x02u, 0xB9u, 0x00u, 0x42u, 0x2Bu, 0x02u, 0x00u, 0x00u, 0x2Bu, 0x00u, 0x54u, 0x80u, 
    0x2Au, 0x00u, 0x01u, 0x20u, 0x40u, 0x33u, 0x00u, 0xCCu, 0x04u, 0x00u, 0x00u, 0x00u, 0x10u, 0x00u, 0x00u, 0xA0u, 
    0x2Au, 0x00u, 0x00u, 0x00u, 0x01u, 0x8Au, 0x2Au, 0x71u, 0x20u, 0x7Bu, 0x00u, 0x80u, 0x00u, 0x04u, 0x00u, 0x00u, 
    0x19u, 0xF8u, 0x07u, 0xF8u, 0x00u, 0x00u, 0x61u, 0x07u, 0x8Au, 0x80u, 0x00u, 0x0Au, 0x00u, 0x0Au, 0x45u, 0x40u, 
    0x52u, 0x06u, 0x00u, 0x00u, 0x04u, 0x4Cu, 0x0Eu, 0x00u, 0x07u, 0xFFu, 0xFFu, 0x0Eu, 0x82u, 0x20u, 0x00u, 0x00u, 
    0x08u, 0x10u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x01u, 0x00u, 0x00u, 0x00u, 0x00u, 0x04u, 0x0Cu, 0x0Cu, 0x04u, 
    0x00u, 0x00u, 0x00u, 0x09u, 0x40u, 0x09u, 0x00u, 0x00u, 0x00u, 0x00u, 0xF0u, 0x10u, 0x00u, 0x13u, 0x10u, 0x13u, 
    0x40u, 0x03u, 0x00u, 0x10u};

/* UDB_UDBPAIR4_UDBSNG1 Address: CYDEV_UDB_UDBPAIR4_UDBSNG1_BASE Size (bytes): 116 */
static const uint8 CYCODE BS_UDB_UDBPAIR4_UDBSNG1_VAL[] = {
    0x00u, 0x00u, 0x00u, 0x00u, 0x26u, 0x00u, 0x19u, 0x00u, 0x12u, 0x00u, 0x2Du, 0x00u, 0x04u, 0x00u, 0x00u, 0x11u, 
    0x02u, 0x08u, 0x00u, 0x06u, 0x04u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x01u, 0x05u, 0x00u, 0x03u, 0x10u, 0x08u, 
    0x15u, 0x00u, 0x2Au, 0x10u, 0x00u, 0x00u, 0x04u, 0x00u, 0x0Eu, 0x0Fu, 0x31u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x07u, 0x00u, 0x10u, 0x3Fu, 0x00u, 0x3Fu, 0x0Cu, 0x00u, 0x00u, 0xA0u, 0x00u, 0xA0u, 0x00u, 0x00u, 0x04u, 
    0x43u, 0x02u, 0x00u, 0x00u, 0x06u, 0x40u, 0xB0u, 0x00u, 0x0Au, 0xFFu, 0xFFu, 0xFFu, 0x13u, 0x5Cu, 0x77u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x02u, 0x74u, 0x00u, 0x00u, 0x00u, 0x0Cu, 0x0Cu, 0x0Cu, 0x0Cu, 
    0x00u, 0x00u, 0x08u, 0x09u, 0x50u, 0xA8u, 0x00u, 0x00u, 0x00u, 0x1Cu, 0x58u, 0xA1u, 0x00u, 0x10u, 0x00u, 0x00u, 
    0x10u, 0x1Du, 0x00u, 0x00u};

/* UDB_UDBPAIR3_UDBSNG0 Address: CYDEV_UDB_UDBPAIR3_UDBSNG0_BASE Size (bytes): 116 */
static const uint8 CYCODE BS_UDB_UDBPAIR3_UDBSNG0_VAL[] = {
    0x3Cu, 0x00u, 0x00u, 0xC0u, 0x00u, 0x80u, 0x10u, 0x40u, 0x0Cu, 0x80u, 0x60u, 0x40u, 0x02u, 0x10u, 0x00u, 0x25u, 
    0x04u, 0x80u, 0x08u, 0x40u, 0x00u, 0x19u, 0x10u, 0x22u, 0x04u, 0x00u, 0x0Bu, 0x00u, 0x5Cu, 0x00u, 0x00u, 0x80u, 
    0x10u, 0x00u, 0x00u, 0x24u, 0x00u, 0x40u, 0x00u, 0x80u, 0x7Cu, 0x25u, 0x00u, 0x1Au, 0x00u, 0x08u, 0x10u, 0x80u, 
    0x01u, 0xC0u, 0x02u, 0x00u, 0x70u, 0x07u, 0x0Cu, 0x38u, 0x00u, 0x00u, 0x80u, 0x82u, 0x00u, 0xA2u, 0x04u, 0x00u, 
    0x43u, 0x02u, 0x00u, 0x00u, 0x01u, 0x00u, 0x00u, 0x00u, 0x00u, 0xFFu, 0xFFu, 0xFFu, 0x12u, 0x5Cu, 0x78u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0xDEu, 0x40u, 0x4Bu, 0x04u, 0x00u, 0x00u, 0x00u, 0x00u, 0x0Cu, 0x0Cu, 0x0Cu, 0x0Cu, 
    0x00u, 0x00u, 0x00u, 0x09u, 0x50u, 0xA8u, 0x00u, 0x00u, 0x00u, 0x1Cu, 0x58u, 0xA1u, 0x00u, 0x10u, 0x00u, 0x00u, 
    0x10u, 0x1Du, 0x00u, 0x00u};

/* UDB_UDBPAIR2_UDBSNG1 Address: CYDEV_UDB_UDBPAIR2_UDBSNG1_BASE Size (bytes): 116 */
static const uint8 CYCODE BS_UDB_UDBPAIR2_UDBSNG1_VAL[] = {
    0x00u, 0x00u, 0x80u, 0x03u, 0x00u, 0xC9u, 0x00u, 0x12u, 0x00u, 0x00u, 0x00u, 0x01u, 0x70u, 0x01u, 0x00u, 0x80u, 
    0x40u, 0x00u, 0x00u, 0x40u, 0x40u, 0xB3u, 0xAAu, 0x4Cu, 0xEAu, 0x01u, 0x00u, 0x00u, 0x00u, 0x00u, 0x15u, 0x40u, 
    0x40u, 0xB1u, 0xAAu, 0x04u, 0x43u, 0x00u, 0x00u, 0x01u, 0x4Cu, 0x05u, 0x00u, 0xBAu, 0x00u, 0x20u, 0x40u, 0x80u, 
    0x03u, 0x7Fu, 0xC0u, 0x80u, 0x0Cu, 0x7Fu, 0x30u, 0x00u, 0xA2u, 0x00u, 0x08u, 0x22u, 0x08u, 0x22u, 0x51u, 0x04u, 
    0x43u, 0x02u, 0x00u, 0x00u, 0x05u, 0xB0u, 0x40u, 0x00u, 0x0Au, 0xFFu, 0xFFu, 0xFFu, 0x13u, 0x5Cu, 0x77u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x0Cu, 0x0Cu, 0x0Cu, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x09u, 0x50u, 0xA8u, 0x00u, 0x00u, 0x00u, 0x1Cu, 0x58u, 0xA1u, 0x00u, 0x10u, 0x00u, 0x00u, 
    0x10u, 0x1Du, 0x00u, 0x00u};

/* UDB_UDBPAIR0_ROUTE Address: CYDEV_UDB_UDBPAIR0_ROUTE_BASE Size (bytes): 144 */
static const uint8 CYCODE BS_UDB_UDBPAIR0_ROUTE_VAL[] = {
    0x14u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0xF5u, 0x7Fu, 0xF3u, 0x11u, 
    0xF3u, 0x11u, 0xFFu, 0xF2u, 0xFFu, 0x2Fu, 0xFFu, 0xFFu, 0x00u, 0x11u, 0x1Fu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0x52u, 
    0xFFu, 0x00u, 0x00u, 0x00u, 0xFFu, 0xFFu, 0xFFu, 0xF7u, 0x5Fu, 0xFFu, 0x2Fu, 0x10u, 0x22u, 0x62u, 0x00u, 0x34u, 
    0x2Fu, 0x62u, 0x73u, 0x00u, 0x11u, 0xFFu, 0xFFu, 0x52u, 0xFFu, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x20u, 0x0Fu, 0x0Fu, 0x27u, 
    0x20u, 0x22u, 0x00u, 0x00u, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 
    0x10u, 0x11u, 0x40u, 0x1Du, 0x11u, 0x00u, 0xDDu, 0x11u, 0x10u, 0x10u, 0x11u, 0x00u, 0xD1u, 0x63u, 0x03u, 0x01u, 
    0x13u, 0x37u, 0x00u, 0x13u, 0x11u, 0x00u, 0x13u, 0x16u, 0x11u, 0x11u, 0x16u, 0x19u, 0x11u, 0x01u, 0x11u, 0x11u, 
    0x19u, 0x11u, 0x11u, 0x91u, 0x41u, 0x11u, 0x21u, 0x11u, 0x16u, 0x11u, 0x11u, 0x13u, 0x81u, 0x11u, 0x11u, 0x11u};

/* UDB_UDBPAIR1_ROUTE Address: CYDEV_UDB_UDBPAIR1_ROUTE_BASE Size (bytes): 144 */
static const uint8 CYCODE BS_UDB_UDBPAIR1_ROUTE_VAL[] = {
    0x00u, 0x02u, 0x00u, 0x00u, 0x00u, 0x00u, 0x04u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x02u, 0x13u, 0x42u, 0x36u, 
    0x26u, 0x33u, 0x35u, 0xF5u, 0xF0u, 0x54u, 0x14u, 0x62u, 0x5Fu, 0x0Fu, 0x04u, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0x52u, 
    0xFFu, 0x00u, 0x00u, 0x00u, 0x24u, 0x45u, 0x7Fu, 0x50u, 0x26u, 0x64u, 0xFFu, 0x3Fu, 0x32u, 0x42u, 0xF3u, 0x32u, 
    0x3Fu, 0x7Fu, 0x37u, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0x52u, 0xFFu, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x01u, 0x40u, 0x02u, 0x05u, 
    0xF6u, 0x00u, 0x00u, 0x00u, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 
    0x10u, 0x13u, 0xF0u, 0x10u, 0x14u, 0x00u, 0x01u, 0x01u, 0x10u, 0x0Du, 0x63u, 0x00u, 0x70u, 0x10u, 0x08u, 0x30u, 
    0x13u, 0x11u, 0x30u, 0x10u, 0x50u, 0x30u, 0x1Fu, 0xC1u, 0x33u, 0x10u, 0x0Fu, 0x01u, 0x31u, 0x0Du, 0x1Cu, 0x25u, 
    0x11u, 0x01u, 0x12u, 0xFBu, 0x02u, 0x12u, 0x15u, 0x1Cu, 0x11u, 0x11u, 0x11u, 0x15u, 0x11u, 0x61u, 0x16u, 0x11u};

/* UDB_UDBPAIR2_ROUTE Address: CYDEV_UDB_UDBPAIR2_ROUTE_BASE Size (bytes): 144 */
static const uint8 CYCODE BS_UDB_UDBPAIR2_ROUTE_VAL[] = {
    0x06u, 0x80u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0xF4u, 0x33u, 0x35u, 0x44u, 
    0x77u, 0x57u, 0xF2u, 0xFFu, 0xF4u, 0xFFu, 0x40u, 0xFFu, 0x32u, 0x63u, 0xFFu, 0xFFu, 0x3Fu, 0x5Fu, 0xFFu, 0x32u, 
    0xFFu, 0x00u, 0x00u, 0x00u, 0xF4u, 0x1Fu, 0x34u, 0x13u, 0x46u, 0x56u, 0x35u, 0x41u, 0x40u, 0x57u, 0x67u, 0x55u, 
    0x3Fu, 0x63u, 0xF6u, 0xFFu, 0x4Fu, 0xFFu, 0xFFu, 0x52u, 0xFFu, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0xF0u, 0x0Fu, 0x00u, 0x00u, 
    0x40u, 0x22u, 0x00u, 0xF0u, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 
    0x10u, 0x11u, 0xF3u, 0x10u, 0x12u, 0x02u, 0x96u, 0x01u, 0x00u, 0x74u, 0xD1u, 0x00u, 0xF7u, 0x03u, 0x03u, 0x16u, 
    0x1Fu, 0x31u, 0x10u, 0x00u, 0x03u, 0x10u, 0x41u, 0x35u, 0x10u, 0x16u, 0x0Fu, 0x76u, 0x11u, 0xB0u, 0xD1u, 0xDFu, 
    0x09u, 0x00u, 0x0Fu, 0xFBu, 0x01u, 0x11u, 0x17u, 0x11u, 0x11u, 0x15u, 0xD5u, 0x1Du, 0x27u, 0x15u, 0x1Fu, 0x11u};

/* UDB_UDBPAIR3_ROUTE Address: CYDEV_UDB_UDBPAIR3_ROUTE_BASE Size (bytes): 144 */
static const uint8 CYCODE BS_UDB_UDBPAIR3_ROUTE_VAL[] = {
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x31u, 0x41u, 0x53u, 0x15u, 
    0x45u, 0x11u, 0xF4u, 0xF3u, 0xFFu, 0xFFu, 0xFFu, 0xF3u, 0x33u, 0x63u, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0x32u, 
    0xFFu, 0x00u, 0x00u, 0x00u, 0x31u, 0x02u, 0x14u, 0x56u, 0xF1u, 0x77u, 0x77u, 0x37u, 0xF0u, 0x56u, 0x73u, 0x45u, 
    0x33u, 0x63u, 0xFFu, 0x03u, 0x15u, 0xF2u, 0xF2u, 0xF2u, 0xF1u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 
    0xB0u, 0x00u, 0xF0u, 0x17u, 0x1Fu, 0x02u, 0x1Fu, 0x60u, 0xA3u, 0xF4u, 0x15u, 0xA0u, 0xF1u, 0x0Fu, 0x00u, 0x31u, 
    0x0Fu, 0xF3u, 0x3Au, 0x20u, 0x51u, 0x3Au, 0x13u, 0xF3u, 0xC0u, 0x10u, 0x4Au, 0x11u, 0x20u, 0x10u, 0x01u, 0xF1u, 
    0x0Fu, 0x00u, 0x0Fu, 0xFFu, 0x41u, 0x11u, 0x41u, 0x4Du, 0x11u, 0x71u, 0x11u, 0x1Fu, 0xFCu, 0x11u, 0x1Fu, 0x1Cu};

/* UDB_UDBPAIR4_ROUTE Address: CYDEV_UDB_UDBPAIR4_ROUTE_BASE Size (bytes): 144 */
static const uint8 CYCODE BS_UDB_UDBPAIR4_ROUTE_VAL[] = {
    0x04u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x74u, 0xF4u, 0x55u, 0x4Fu, 
    0xF3u, 0x5Fu, 0x55u, 0x27u, 0x31u, 0x63u, 0x71u, 0x61u, 0x3Fu, 0x63u, 0x0Fu, 0x5Fu, 0xFFu, 0xFFu, 0xFFu, 0x12u, 
    0xFFu, 0x00u, 0x00u, 0x00u, 0x0Fu, 0x64u, 0x31u, 0x70u, 0x77u, 0xF6u, 0x2Fu, 0x1Fu, 0x25u, 0x4Fu, 0xF7u, 0x3Fu, 
    0x32u, 0x63u, 0x3Fu, 0x75u, 0xF4u, 0x1Fu, 0xFFu, 0x12u, 0xF3u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x0Fu, 0x00u, 0x00u, 
    0x04u, 0x00u, 0x00u, 0x00u, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 
    0x70u, 0x30u, 0x26u, 0x14u, 0x2Fu, 0x0Cu, 0xA5u, 0x10u, 0x18u, 0x11u, 0x0Fu, 0xF0u, 0xF7u, 0x3Fu, 0x00u, 0x11u, 
    0x0Fu, 0xFFu, 0xF8u, 0x16u, 0xF7u, 0x1Fu, 0x10u, 0x11u, 0xF4u, 0x13u, 0xFFu, 0x67u, 0x13u, 0x15u, 0x07u, 0xF1u, 
    0x3Fu, 0xCDu, 0x2Fu, 0x4Fu, 0xF6u, 0x12u, 0x12u, 0x11u, 0x11u, 0x19u, 0x11u, 0x21u, 0xFFu, 0x11u, 0x1Fu, 0x11u};

/* UDB_UDBPAIR5_ROUTE Address: CYDEV_UDB_UDBPAIR5_ROUTE_BASE Size (bytes): 144 */
static const uint8 CYCODE BS_UDB_UDBPAIR5_ROUTE_VAL[] = {
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x31u, 0x32u, 0xF3u, 0xF4u, 
    0xFFu, 0x45u, 0xFFu, 0x43u, 0x12u, 0x35u, 0x33u, 0xFFu, 0x3Fu, 0x5Fu, 0x34u, 0xF0u, 0xFFu, 0x2Fu, 0xFFu, 0x12u, 
    0xFFu, 0x00u, 0x00u, 0x00u, 0xFFu, 0xFFu, 0xFFu, 0xF3u, 0xF3u, 0xFFu, 0x44u, 0x1Fu, 0xF2u, 0x04u, 0x01u, 0x2Fu, 
    0x30u, 0x2Fu, 0x3Fu, 0xFFu, 0xF0u, 0x10u, 0xFFu, 0x12u, 0xF3u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 
    0xF5u, 0x17u, 0x11u, 0x16u, 0x11u, 0x2Cu, 0x1Fu, 0x17u, 0x9Fu, 0x41u, 0x6Fu, 0xF4u, 0x1Fu, 0x33u, 0xCCu, 0x11u, 
    0x63u, 0x1Fu, 0x29u, 0x71u, 0xF5u, 0x2Fu, 0x73u, 0x13u, 0xF1u, 0x13u, 0x11u, 0x1Au, 0x11u, 0x31u, 0x6Fu, 0x11u, 
    0x11u, 0x1Fu, 0x11u, 0x1Au, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u};

/* UDB_DSI0 Address: CYDEV_UDB_DSI0_BASE Size (bytes): 124 */
static const uint8 CYCODE BS_UDB_DSI0_VAL[] = {
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u};

/* UDB_DSI1 Address: CYDEV_UDB_DSI1_BASE Size (bytes): 124 */
static const uint8 CYCODE BS_UDB_DSI1_VAL[] = {
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u};

/* UDB_DSI2 Address: CYDEV_UDB_DSI2_BASE Size (bytes): 124 */
static const uint8 CYCODE BS_UDB_DSI2_VAL[] = {
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u};

/* UDB_DSI3 Address: CYDEV_UDB_DSI3_BASE Size (bytes): 124 */
static const uint8 CYCODE BS_UDB_DSI3_VAL[] = {
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x15u, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0xF1u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u};

/* UDB_DSI4 Address: CYDEV_UDB_DSI4_BASE Size (bytes): 124 */
static const uint8 CYCODE BS_UDB_DSI4_VAL[] = {
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x0Fu, 0x00u, 0x00u, 0x00u, 
    0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0xD1u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u};

/* UDB_DSI5 Address: CYDEV_UDB_DSI5_BASE Size (bytes): 124 */
static const uint8 CYCODE BS_UDB_DSI5_VAL[] = {
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u};

/* UDB_DSI6 Address: CYDEV_UDB_DSI6_BASE Size (bytes): 124 */
static const uint8 CYCODE BS_UDB_DSI6_VAL[] = {
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x00u, 0x09u, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x0Eu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0xF0u, 0x00u, 0x03u, 0xFFu, 0xF0u, 0xFFu, 0x00u, 0x00u, 
    0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0x11u, 0xD1u, 0x01u, 0x11u, 
    0x11u, 0x11u, 0x1Du, 0x11u, 0x11u, 0x1Du, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x10u, 0xD1u, 0x11u, 0x11u, 0x11u, 0xD1u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0xD1u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u};

/* UDB_DSI7 Address: CYDEV_UDB_DSI7_BASE Size (bytes): 124 */
static const uint8 CYCODE BS_UDB_DSI7_VAL[] = {
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x02u, 0x08u, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x08u, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x16u, 0x04u, 0x12u, 0x10u, 0x0Du, 0x1Fu, 0x1Fu, 0x10u, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x0Fu, 0xF0u, 0x00u, 0xFFu, 0x0Fu, 0x00u, 0x00u, 0x00u, 
    0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0x1Du, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x0Fu, 0x11u, 0x11u, 0xD1u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0xD1u, 0x11u, 
    0x11u, 0xF1u, 0x11u, 0x11u, 0x21u, 0xF1u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0xD1u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0xD1u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u};

/* UDB_DSI8 Address: CYDEV_UDB_DSI8_BASE Size (bytes): 124 */
static const uint8 CYCODE BS_UDB_DSI8_VAL[] = {
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x50u, 0x04u, 0x00u, 0x00u, 0x00u, 0x0Fu, 0x00u, 0x20u, 
    0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0xD1u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0xF1u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x1Fu, 0xF1u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x1Fu, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u};

/* UDB_DSI9 Address: CYDEV_UDB_DSI9_BASE Size (bytes): 124 */
static const uint8 CYCODE BS_UDB_DSI9_VAL[] = {
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x1Fu, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u};

/* UDB_DSI10 Address: CYDEV_UDB_DSI10_BASE Size (bytes): 124 */
static const uint8 CYCODE BS_UDB_DSI10_VAL[] = {
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x03u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x1Fu, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u};

/* UDB_DSI11 Address: CYDEV_UDB_DSI11_BASE Size (bytes): 124 */
static const uint8 CYCODE BS_UDB_DSI11_VAL[] = {
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 
    0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x1Fu, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 0x00u, 
    0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0xFFu, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 
    0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u, 0x11u};

static const cfg_memcpy_t CYCODE cfg_memcpy_list [] = {
    /* dest, src, size */
    {(void CYFAR *)(CYDEV_UDB_UDBPAIR5_UDBSNG1_BASE), BS_UDB_UDBPAIR5_UDBSNG1_VAL, 116u},
    {(void CYFAR *)(CYDEV_UDB_UDBPAIR1_UDBSNG1_BASE), BS_UDB_UDBPAIR1_UDBSNG1_VAL, 116u},
    {(void CYFAR *)(CYDEV_UDB_UDBPAIR4_UDBSNG1_BASE), BS_UDB_UDBPAIR4_UDBSNG1_VAL, 116u},
    {(void CYFAR *)(CYDEV_UDB_UDBPAIR3_UDBSNG0_BASE), BS_UDB_UDBPAIR3_UDBSNG0_VAL, 116u},
    {(void CYFAR *)(CYDEV_UDB_UDBPAIR2_UDBSNG1_BASE), BS_UDB_UDBPAIR2_UDBSNG1_VAL, 116u},
    {(void CYFAR *)(CYDEV_UDB_UDBPAIR0_ROUTE_BASE), BS_UDB_UDBPAIR0_ROUTE_VAL, 144u},
    {(void CYFAR *)(CYDEV_UDB_UDBPAIR1_ROUTE_BASE), BS_UDB_UDBPAIR1_ROUTE_VAL, 144u},
    {(void CYFAR *)(CYDEV_UDB_UDBPAIR2_ROUTE_BASE), BS_UDB_UDBPAIR2_ROUTE_VAL, 144u},
    {(void CYFAR *)(CYDEV_UDB_UDBPAIR3_ROUTE_BASE), BS_UDB_UDBPAIR3_ROUTE_VAL, 144u},
    {(void CYFAR *)(CYDEV_UDB_UDBPAIR4_ROUTE_BASE), BS_UDB_UDBPAIR4_ROUTE_VAL, 144u},
    {(void CYFAR *)(CYDEV_UDB_UDBPAIR5_ROUTE_BASE), BS_UDB_UDBPAIR5_ROUTE_VAL, 144u},
    {(void CYFAR *)(CYDEV_UDB_DSI0_BASE), BS_UDB_DSI0_VAL, 124u},
    {(void CYFAR *)(CYDEV_UDB_DSI1_BASE), BS_UDB_DSI1_VAL, 124u},
    {(void CYFAR *)(CYDEV_UDB_DSI2_BASE), BS_UDB_DSI2_VAL, 124u},
    {(void CYFAR *)(CYDEV_UDB_DSI3_BASE), BS_UDB_DSI3_VAL, 124u},
    {(void CYFAR *)(CYDEV_UDB_DSI4_BASE), BS_UDB_DSI4_VAL, 124u},
    {(void CYFAR *)(CYDEV_UDB_DSI5_BASE), BS_UDB_DSI5_VAL, 124u},
    {(void CYFAR *)(CYDEV_UDB_DSI6_BASE), BS_UDB_DSI6_VAL, 124u},
    {(void CYFAR *)(CYDEV_UDB_DSI7_BASE), BS_UDB_DSI7_VAL, 124u},
    {(void CYFAR *)(CYDEV_UDB_DSI8_BASE), BS_UDB_DSI8_VAL, 124u},
    {(void CYFAR *)(CYDEV_UDB_DSI9_BASE), BS_UDB_DSI9_VAL, 124u},
    {(void CYFAR *)(CYDEV_UDB_DSI10_BASE), BS_UDB_DSI10_VAL, 124u},
    {(void CYFAR *)(CYDEV_UDB_DSI11_BASE), BS_UDB_DSI11_VAL, 124u},
};

void SDIO_Host_Config_TriggerMuxes(void)
{
    /* Connect UDB to DMA */
    Cy_TrigMux_Connect(TRIG0_IN_TR_GROUP14_OUTPUT1, TRIG0_OUT_CPUSS_DW0_TR_IN1, false, TRIGGER_TYPE_LEVEL);
    Cy_TrigMux_Connect(TRIG0_IN_TR_GROUP14_OUTPUT4, TRIG0_OUT_CPUSS_DW0_TR_IN0, false, TRIGGER_TYPE_LEVEL);
    Cy_TrigMux_Connect(TRIG1_IN_TR_GROUP14_OUTPUT0, TRIG1_OUT_CPUSS_DW1_TR_IN1, false, TRIGGER_TYPE_LEVEL);
    Cy_TrigMux_Connect(TRIG1_IN_TR_GROUP14_OUTPUT5, TRIG1_OUT_CPUSS_DW1_TR_IN3, false, TRIGGER_TYPE_LEVEL);

    Cy_TrigMux_Connect(TRIG14_IN_UDB_TR_UDB0, TRIG14_OUT_TR_GROUP1_INPUT43, false, TRIGGER_TYPE_LEVEL);
    Cy_TrigMux_Connect(TRIG14_IN_UDB_TR_UDB1, TRIG14_OUT_TR_GROUP0_INPUT44, false, TRIGGER_TYPE_LEVEL);
    Cy_TrigMux_Connect(TRIG14_IN_UDB_TR_UDB3, TRIG14_OUT_TR_GROUP0_INPUT47, false, TRIGGER_TYPE_LEVEL);
    Cy_TrigMux_Connect(TRIG14_IN_UDB_TR_UDB7, TRIG14_OUT_TR_GROUP1_INPUT48, false, TRIGGER_TYPE_LEVEL);
}

void SDIO_Host_Config_UDBs(void)
{
    size_t i;

    /* Power on the UDB array */
    CY_SET_REG32(0x402101F0u, 0x05FA0003u);

    /* Zero out critical memory blocks before beginning configuration */
    for (i = 0u; i < (sizeof(cfg_memset_list)/sizeof(cfg_memset_list[0])); i++)
    {
        const cfg_memset_t *ms = &cfg_memset_list[i];
        CYMEMZERO(ms->address, ms->size);
    }

    /* Copy device configuration data into registers */
    for (i = 0u; i < (sizeof(cfg_memcpy_list)/sizeof(cfg_memcpy_list[0])); i++)
    {
        const cfg_memcpy_t *mc = &cfg_memcpy_list[i];
        CYCONFIGCPYCODE(mc->dest, mc->src, mc->size);
    }

    cfg_write_bytes32(cy_cfg_addr_table, cy_cfg_data_table);

    /* UDB_INT_CFG Starting address: CYDEV_UDB_UDBIF_INT_CLK_CTL */
    CY_SET_REG32((void *)(CYREG_UDB_UDBIF_INT_CLK_CTL), 0x00000001u);

    /* UDB_UDBPAIR0_UDBSNG0_RC_CFG0 Starting address: CYDEV_UDB_UDBPAIR0_UDBSNG0_RC_CFG0 */
    CY_SET_REG32((void *)(CYREG_UDB_UDBPAIR0_UDBSNG0_RC_CFG0), 0x004C404Cu);

    /* UDB_UDBPAIR0_UDBSNG1_RC_CFG0 Starting address: CYDEV_UDB_UDBPAIR0_UDBSNG1_RC_CFG0 */
    CY_SET_REG32((void *)(CYREG_UDB_UDBPAIR0_UDBSNG1_RC_CFG0), 0x044C4C44u);

    /* UDB_UDBPAIR1_UDBSNG0_RC_CFG0 Starting address: CYDEV_UDB_UDBPAIR1_UDBSNG0_RC_CFG0 */
    CY_SET_REG32((void *)(CYREG_UDB_UDBPAIR1_UDBSNG0_RC_CFG0), 0x004C444Cu);

    /* UDB_UDBPAIR1_UDBSNG1_RC_CFG0 Starting address: CYDEV_UDB_UDBPAIR1_UDBSNG1_RC_CFG0 */
    CY_SET_REG32((void *)(CYREG_UDB_UDBPAIR1_UDBSNG1_RC_CFG0), 0x044C4C44u);

    /* UDB_UDBPAIR2_UDBSNG0_RC_CFG0 Starting address: CYDEV_UDB_UDBPAIR2_UDBSNG0_RC_CFG0 */
    CY_SET_REG32((void *)(CYREG_UDB_UDBPAIR2_UDBSNG0_RC_CFG0), 0x4C4C404Cu);

    /* UDB_UDBPAIR2_UDBSNG1_RC_CFG0 Starting address: CYDEV_UDB_UDBPAIR2_UDBSNG1_RC_CFG0 */
    CY_SET_REG32((void *)(CYREG_UDB_UDBPAIR2_UDBSNG1_RC_CFG0), 0x004C4C4Cu);

    /* UDB_UDBPAIR3_UDBSNG0_RC_CFG0 Starting address: CYDEV_UDB_UDBPAIR3_UDBSNG0_RC_CFG0 */
    CY_SET_REG32((void *)(CYREG_UDB_UDBPAIR3_UDBSNG0_RC_CFG0), 0x0C8C8C8Cu);

    /* UDB_UDBPAIR3_UDBSNG1_RC_CFG0 Starting address: CYDEV_UDB_UDBPAIR3_UDBSNG1_RC_CFG0 */
    CY_SET_REG32((void *)(CYREG_UDB_UDBPAIR3_UDBSNG1_RC_CFG0), 0x004C404Cu);

    /* UDB_UDBPAIR4_UDBSNG0_RC_CFG0 Starting address: CYDEV_UDB_UDBPAIR4_UDBSNG0_RC_CFG0 */
    CY_SET_REG32((void *)(CYREG_UDB_UDBPAIR4_UDBSNG0_RC_CFG0), 0x004C444Cu);

    /* UDB_UDBPAIR4_UDBSNG1_RC_CFG0 Starting address: CYDEV_UDB_UDBPAIR4_UDBSNG1_RC_CFG0 */
    CY_SET_REG32((void *)(CYREG_UDB_UDBPAIR4_UDBSNG1_RC_CFG0), 0x4C4C4C4Cu);

    /* UDB_UDBPAIR5_UDBSNG0_RC_CFG0 Starting address: CYDEV_UDB_UDBPAIR5_UDBSNG0_RC_CFG0 */
    CY_SET_REG32((void *)(CYREG_UDB_UDBPAIR5_UDBSNG0_RC_CFG0), 0x4C4C0400u);

    /* UDB_UDBPAIR5_UDBSNG1_RC_CFG0 Starting address: CYDEV_UDB_UDBPAIR5_UDBSNG1_RC_CFG0 */
    CY_SET_REG32((void *)(CYREG_UDB_UDBPAIR5_UDBSNG1_RC_CFG0), 0x4C4C4C4Cu);

    /* Enable UDB array and digital routing */
    CY_SET_REG32((void *)0x40347900u, CY_GET_REG32((void *)0x40347900u) | 0x106u);
}

#if defined(__cplusplus)
}
#endif

/* [] END OF FILE */
