# Copyright 2016 Google Inc. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


import os

import unittest

from google.cloud import translate


ENV_VAR = 'GOOGLE_CLOUD_TESTS_API_KEY'


class Config(object):
    """Run-time configuration to be modified at set-up.

    This is a mutable stand-in to allow test set-up to modify
    global state.
    """
    CLIENT = None


def setUpModule():
    api_key = os.getenv(ENV_VAR)
    Config.CLIENT = translate.Client(api_key=api_key)


class TestTranslate(unittest.TestCase):

    def test_get_languages(self):
        result = Config.CLIENT.get_languages()
        # There are **many** more than 10 languages.
        self.assertGreater(len(result), 10)

        lang_map = {val['language']: val['name'] for val in result}
        self.assertEqual(lang_map['en'], 'English')
        self.assertEqual(lang_map['ja'], 'Japanese')
        self.assertEqual(lang_map['lv'], 'Latvian')
        self.assertEqual(lang_map['zu'], 'Zulu')

    def test_detect_language(self):
        values = ['takoy', u'fa\xe7ade', 's\'il vous plait']
        detections = Config.CLIENT.detect_language(values)
        self.assertEqual(len(values), len(detections))
        self.assertEqual(detections[0]['language'], 'ru')
        self.assertEqual(detections[1]['language'], 'fr')
        self.assertEqual(detections[2]['language'], 'fr')

    def test_translate(self):
        values = ['hvala ti', 'dankon',
                  'Me llamo Jeff', 'My name is Jeff']
        translations = Config.CLIENT.translate(values,
                                               target_language='de')
        self.assertEqual(len(values), len(translations))

        self.assertEqual(
            translations[0]['detectedSourceLanguage'], 'hr')
        self.assertEqual(
            translations[0]['translatedText'], 'danke')

        self.assertEqual(
            translations[1]['detectedSourceLanguage'], 'eo')
        self.assertEqual(
            translations[1]['translatedText'], 'dank')

        self.assertEqual(
            translations[2]['detectedSourceLanguage'], 'es')
        self.assertEqual(
            translations[2]['translatedText'], 'Mein Name ist Jeff')

        self.assertEqual(
            translations[3]['detectedSourceLanguage'], 'en')
        self.assertEqual(
            translations[3]['translatedText'], 'Mein Name ist Jeff')
