/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at <http://mozilla.org/MPL/2.0/>. */

// @flow

import type {
  SourcePacket,
  ResumedPacket,
  PausedPacket,
  ThreadClient,
  Actions
} from "./types";

import { createPause, createSource } from "./create";
import sourceQueue from "../../utils/source-queue";
import { features } from "../../utils/prefs";

const CALL_STACK_PAGE_SIZE = 1000;

type Dependencies = {
  threadClient: ThreadClient,
  actions: Actions,
  supportsWasm: boolean
};

let threadClient: ThreadClient;
let actions: Actions;
let supportsWasm: boolean;
let isInterrupted: boolean;

function setupEvents(dependencies: Dependencies) {
  threadClient = dependencies.threadClient;
  actions = dependencies.actions;
  supportsWasm = dependencies.supportsWasm;
  sourceQueue.initialize({ actions, supportsWasm, createSource });

  if (threadClient) {
    Object.keys(clientEvents).forEach(eventName => {
      threadClient.addListener(eventName, clientEvents[eventName]);
    });

    if (threadClient._parent) {
      threadClient._parent.addListener("workerListChanged", workerListChanged);
    }
  }
}

async function paused(_: "paused", packet: PausedPacket) {
  // If paused by an explicit interrupt, which are generated by the
  // slow script dialog and internal events such as setting
  // breakpoints, ignore the event.
  const { why } = packet;
  if (why.type === "interrupted" && !packet.why.onNext) {
    isInterrupted = true;
    return;
  }

  // Eagerly fetch the frames
  const response = await threadClient.getFrames(0, CALL_STACK_PAGE_SIZE);

  if (why.type != "alreadyPaused") {
    const pause = createPause(packet, response);
    await sourceQueue.flush();
    actions.paused(pause);
  }
}

function resumed(_: "resumed", packet: ResumedPacket) {
  // NOTE: the client suppresses resumed events while interrupted
  // to prevent unintentional behavior.
  // see [client docs](../README.md#interrupted) for more information.
  if (isInterrupted) {
    isInterrupted = false;
    return;
  }

  actions.resumed(packet);
}

function newSource(_: "newSource", { source }: SourcePacket) {
  sourceQueue.queue(source);

  if (features.eventListeners) {
    actions.fetchEventListeners();
  }
}

function workerListChanged() {
  actions.updateWorkers();
}

const clientEvents = {
  paused,
  resumed,
  newSource
};

export { setupEvents, clientEvents };
