import {
	CUSTOM_ELEMENTS_SCHEMA,
	ChangeDetectionStrategy,
	Component,
	Directive,
	ElementRef,
	computed,
	effect,
	inject,
	input,
	viewChild,
} from '@angular/core';
import { NgtArgs } from 'angular-three';
import { gltfResource } from 'angular-three-soba/loaders';
import { NgtsAnimation, animations } from 'angular-three-soba/misc';
import { matcapTextureResource } from 'angular-three-soba/staging';
import { Bone, Group, MeshStandardMaterial, Object3D, SRGBColorSpace, SkinnedMesh } from 'three';
import { GLTF } from 'three-stdlib';
import { select, storyDecorators, storyObject } from '../setup-canvas';

type BotGLTF = GLTF & {
	nodes: { 'Y-Bot': Object3D; YB_Body: SkinnedMesh; YB_Joints: SkinnedMesh; mixamorigHips: Bone };
	materials: { YB_Body: MeshStandardMaterial; YB_Joints: MeshStandardMaterial };
};

@Directive({ selector: '[animations]' })
export class BotAnimations {
	animations = input.required<NgtsAnimation>();
	animation = input('Strut');
	referenceRef = input.required<ElementRef<Bone> | undefined>();

	constructor() {
		const groupRef = inject<ElementRef<Group>>(ElementRef);
		const host = computed(() => (this.referenceRef() ? groupRef : null));

		// NOTE: the consumer controls the timing of injectAnimations. It's not afterNextRender anymore
		//  but when the reference is resolved which in this particular case, it is the Bone mixamorigHips
		//  that the animations are referring to.
		const animationsApi = animations(this.animations, host);
		effect((onCleanup) => {
			if (animationsApi.isReady) {
				const actionName = this.animation();
				animationsApi.actions[actionName].reset().fadeIn(0.5).play();
				onCleanup(() => {
					animationsApi.actions[actionName].fadeOut(0.5);
				});
			}
		});
	}
}

@Component({
	template: `
		<ngt-group [position]="[0, -1, 0]">
			<ngt-grid-helper *args="[10, 20]" />
			@if (gltf.value(); as gltf) {
				<ngt-group [dispose]="null" [animations]="gltf" [animation]="animation()" [referenceRef]="boneRef()">
					<ngt-group [rotation]="[Math.PI / 2, 0, 0]" [scale]="0.01">
						<ngt-primitive #bone *args="[gltf.nodes.mixamorigHips]" />
						<ngt-skinned-mesh
							[geometry]="gltf.nodes.YB_Body.geometry"
							[skeleton]="gltf.nodes.YB_Body.skeleton"
						>
							<ngt-mesh-matcap-material [matcap]="matcapBody.resource.value()" />
						</ngt-skinned-mesh>
						<ngt-skinned-mesh
							[geometry]="gltf.nodes.YB_Joints.geometry"
							[skeleton]="gltf.nodes.YB_Joints.skeleton"
						>
							<ngt-mesh-matcap-material [matcap]="matcapJoints.resource.value()" />
						</ngt-skinned-mesh>
					</ngt-group>
				</ngt-group>
			}
		</ngt-group>
	`,
	imports: [NgtArgs, BotAnimations],
	schemas: [CUSTOM_ELEMENTS_SCHEMA],
	changeDetection: ChangeDetectionStrategy.OnPush,
})
class DefaultAnimationsStory {
	Math = Math;

	animation = input('Strut');

	gltf = gltfResource<BotGLTF>(() => './ybot.glb');
	matcapBody = matcapTextureResource(() => '293534_B2BFC5_738289_8A9AA7', {
		onLoad: (texture) => (texture.colorSpace = SRGBColorSpace),
	});
	matcapJoints = matcapTextureResource(() => '3A2412_A78B5F_705434_836C47', {
		onLoad: (texture) => (texture.colorSpace = SRGBColorSpace),
	});

	boneRef = viewChild<ElementRef<Bone>>('bone');
}

export default {
	title: 'Misc/injectAnimations',
	decorators: storyDecorators(),
};

export const Default = storyObject(DefaultAnimationsStory, {
	camera: { position: [0, 0, 3] },
	argsOptions: {
		animation: select('Strut', { options: ['Strut', 'Dance', 'Idle'] }),
	},
});
