import {
	CUSTOM_ELEMENTS_SCHEMA,
	ChangeDetectionStrategy,
	Component,
	ElementRef,
	computed,
	input,
	viewChild,
} from '@angular/core';
import { NgtArgs, NgtThreeElements, beforeRender, extend, injectStore, omit, pick } from 'angular-three';
import { mergeInputs } from 'ngxtension/inject-inputs';
import * as THREE from 'three';
import { Group, Mesh, MeshBasicMaterial, OrthographicCamera } from 'three';
import { HorizontalBlurShader, VerticalBlurShader } from 'three-stdlib';

/**
 * Configuration options for the NgtsContactShadows component.
 * Extends the standard ngt-group element options (excluding scale which has custom handling).
 */
export interface NgtsContactShadowsOptions extends Partial<Omit<NgtThreeElements['ngt-group'], 'scale'>> {
	/**
	 * Opacity of the shadow.
	 * @default 1
	 */
	opacity: number;
	/**
	 * Width of the shadow plane (before scaling).
	 * @default 1
	 */
	width: number;
	/**
	 * Height of the shadow plane (before scaling).
	 * @default 1
	 */
	height: number;
	/**
	 * Blur amount for the shadow. Higher values create softer shadows.
	 * @default 1
	 */
	blur: number;
	/**
	 * Near clipping plane for the shadow camera.
	 * @default 0
	 */
	near: number;
	/**
	 * Far clipping plane for the shadow camera.
	 * @default 10
	 */
	far: number;
	/**
	 * When enabled, applies an additional blur pass for smoother shadows.
	 * @default true
	 */
	smooth: boolean;
	/**
	 * Resolution of the shadow render target.
	 * @default 512
	 */
	resolution: number;
	/**
	 * Number of frames to render. Use Infinity for continuous updates.
	 * @default Infinity
	 */
	frames: number;
	/**
	 * Scale of the shadow plane. Can be a single number or [x, y] tuple.
	 * @default 10
	 */
	scale: number | [x: number, y: number];
	/**
	 * Color of the shadow.
	 * @default '#000000'
	 */
	color: THREE.ColorRepresentation;
	/**
	 * Whether the shadow writes to the depth buffer.
	 * @default false
	 */
	depthWrite: boolean;
}

const defaultOptions: NgtsContactShadowsOptions = {
	scale: 10,
	frames: Infinity,
	opacity: 1,
	width: 1,
	height: 1,
	blur: 1,
	near: 0,
	far: 10,
	resolution: 512,
	smooth: true,
	color: '#000000',
	depthWrite: false,
};

/**
 * A component that renders soft contact shadows on a ground plane.
 * Uses a depth-based technique to render shadows from objects onto
 * a horizontal plane, creating realistic grounding effects.
 *
 * The shadows are rendered using an orthographic camera from above,
 * with optional blur passes for softness.
 *
 * @example
 * ```html
 * <ngts-contact-shadows
 *   [options]="{ opacity: 0.5, scale: 10, blur: 2, far: 4, resolution: 256 }"
 *   [position]="[0, -0.5, 0]"
 * />
 * ```
 */
@Component({
	selector: 'ngts-contact-shadows',
	template: `
		<ngt-group #contactShadows [rotation]="[Math.PI / 2, 0, 0]" [parameters]="parameters()">
			<ngt-mesh
				[scale]="[1, -1, 1]"
				[rotation]="[-Math.PI / 2, 0, 0]"
				[renderOrder]="renderOrder() ?? 0"
				[geometry]="planeGeometry()"
			>
				<ngt-mesh-basic-material
					transparent
					[opacity]="opacity()"
					[depthWrite]="depthWrite()"
					[map]="texture()"
				/>
			</ngt-mesh>
			<ngt-orthographic-camera *args="cameraArgs()" #shadowsCamera />
		</ngt-group>
	`,
	imports: [NgtArgs],
	schemas: [CUSTOM_ELEMENTS_SCHEMA],
	changeDetection: ChangeDetectionStrategy.OnPush,
})
export class NgtsContactShadows {
	protected readonly Math = Math;

	options = input(defaultOptions, { transform: mergeInputs(defaultOptions) });
	protected parameters = omit(this.options, [
		'scale',
		'frames',
		'opacity',
		'width',
		'height',
		'blur',
		'near',
		'far',
		'resolution',
		'smooth',
		'color',
		'depthWrite',
		'renderOrder',
	]);

	contactShadowsRef = viewChild.required<ElementRef<THREE.Group>>('contactShadows');
	private shadowsCameraRef = viewChild<ElementRef<THREE.OrthographicCamera>>('shadowsCamera');

	private store = injectStore();

	private width = pick(this.options, 'width');
	private height = pick(this.options, 'height');
	private scale = pick(this.options, 'scale');

	private scaledWidth = computed(() => {
		const [width, scale] = [this.width(), this.scale()];
		return width * (Array.isArray(scale) ? scale[0] : scale);
	});

	private scaledHeight = computed(() => {
		const [height, scale] = [this.height(), this.scale()];
		return height * (Array.isArray(scale) ? scale[1] : scale);
	});

	private resolution = pick(this.options, 'resolution');
	private color = pick(this.options, 'color');
	private near = pick(this.options, 'near');
	private far = pick(this.options, 'far');
	private smooth = pick(this.options, 'smooth');
	private frames = pick(this.options, 'frames');
	private blur = pick(this.options, 'blur');

	private renderTarget = computed(() => this.createRenderTarget(this.resolution()));
	private renderTargetBlur = computed(() => this.createRenderTarget(this.resolution()));

	protected planeGeometry = computed(() =>
		new THREE.PlaneGeometry(this.scaledWidth(), this.scaledHeight()).rotateX(Math.PI / 2),
	);
	private blurPlane = computed(() => new THREE.Mesh(this.planeGeometry()));
	private depthMaterial = computed(() => {
		const color = new THREE.Color(this.color());
		const material = new THREE.MeshDepthMaterial();
		material.depthTest = material.depthWrite = false;
		material.onBeforeCompile = (shader) => {
			shader.uniforms = { ...shader.uniforms, ucolor: { value: color } };
			shader.fragmentShader = shader.fragmentShader.replace(
				`void main() {`, //
				`uniform vec3 ucolor;
         void main() {
        `,
			);
			shader.fragmentShader = shader.fragmentShader.replace(
				'vec4( vec3( 1.0 - fragCoordZ ), opacity );',
				// Colorize the shadow, multiply by the falloff so that the center can remain darker
				'vec4( ucolor * fragCoordZ * 2.0, ( 1.0 - fragCoordZ ) * 1.0 );',
			);
		};
		return material;
	});

	private horizontalBlurMaterial = new THREE.ShaderMaterial({ ...HorizontalBlurShader, depthTest: false });
	private verticalBlurMaterial = new THREE.ShaderMaterial({ ...VerticalBlurShader, depthTest: false });

	protected renderOrder = pick(this.options, 'renderOrder');
	protected opacity = pick(this.options, 'opacity');
	protected depthWrite = pick(this.options, 'depthWrite');
	protected texture = pick(this.renderTarget, 'texture');
	protected cameraArgs = computed(() => {
		const [width, height, near, far] = [this.scaledWidth(), this.scaledHeight(), this.near(), this.far()];
		return [-width / 2, width / 2, height / 2, -height / 2, near, far];
	});

	constructor() {
		extend({ Group, Mesh, MeshBasicMaterial, OrthographicCamera });

		let count = 0;
		beforeRender(() => {
			const shadowsCamera = this.shadowsCameraRef()?.nativeElement;
			if (!shadowsCamera) return;

			const frames = this.frames();
			if (frames === Infinity || count < frames) {
				this.renderShadows();
				count++;
			}
		});
	}

	private renderShadows() {
		const shadowsCamera = this.shadowsCameraRef()?.nativeElement;
		if (!shadowsCamera) return;

		const [blur, smooth, gl, scene, contactShadows, depthMaterial, renderTarget] = [
			this.blur(),
			this.smooth(),
			this.store.snapshot.gl,
			this.store.snapshot.scene,
			this.contactShadowsRef().nativeElement,
			this.depthMaterial(),
			this.renderTarget(),
		];

		const initialBackground = scene.background;
		const initialOverrideMaterial = scene.overrideMaterial;
		const initialClearAlpha = gl.getClearAlpha();

		contactShadows.visible = false;
		scene.background = null;
		scene.overrideMaterial = depthMaterial;
		gl.setClearAlpha(0);

		// render to the render target to get the depths
		gl.setRenderTarget(renderTarget);
		gl.render(scene, shadowsCamera);

		this.blurShadows(blur);
		if (smooth) this.blurShadows(blur * 0.4);

		// reset
		gl.setRenderTarget(null);

		contactShadows.visible = true;
		scene.overrideMaterial = initialOverrideMaterial;
		scene.background = initialBackground;
		gl.setClearAlpha(initialClearAlpha);
	}

	private blurShadows(blur: number) {
		const shadowsCamera = this.shadowsCameraRef()?.nativeElement;
		if (!shadowsCamera) return;

		const [blurPlane, horizontalBlurMaterial, verticalBlurMaterial, renderTargetBlur, renderTarget, gl] = [
			this.blurPlane(),
			this.horizontalBlurMaterial,
			this.verticalBlurMaterial,
			this.renderTargetBlur(),
			this.renderTarget(),
			this.store.snapshot.gl,
		];

		blurPlane.visible = true;
		blurPlane.material = horizontalBlurMaterial;
		horizontalBlurMaterial.uniforms['tDiffuse'].value = renderTarget.texture;
		horizontalBlurMaterial.uniforms['h'].value = blur / 256;
		gl.setRenderTarget(renderTargetBlur);
		gl.render(blurPlane, shadowsCamera);

		blurPlane.material = verticalBlurMaterial;
		verticalBlurMaterial.uniforms['tDiffuse'].value = renderTargetBlur.texture;
		verticalBlurMaterial.uniforms['v'].value = blur / 256;
		gl.setRenderTarget(renderTarget);
		gl.render(blurPlane, shadowsCamera);
		blurPlane.visible = false;
	}

	private createRenderTarget(resolution: number) {
		const renderTarget = new THREE.WebGLRenderTarget(resolution, resolution);
		renderTarget.texture.generateMipmaps = false;
		return renderTarget;
	}
}
