import {
	ChangeDetectionStrategy,
	Component,
	CUSTOM_ELEMENTS_SCHEMA,
	effect,
	ElementRef,
	input,
	viewChild,
} from '@angular/core';
import { getInstanceState, NgtVector3 } from 'angular-three';
import { gltfResource } from 'angular-three-soba/loaders';
import { NgtsMeshTransmissionMaterial } from 'angular-three-soba/materials';
import { mask } from 'angular-three-soba/staging';
import { Group, Mesh } from 'three';
import { GLTF } from 'three-stdlib';

import shapesGLB from './shapes-transformed.glb';

interface ShapesGLTF extends GLTF {
	nodes: { Cube: Mesh };
}

@Component({
	selector: 'app-tank',
	template: `
		@if (gltf.value(); as gltf) {
			@let nodes = gltf.nodes;

			<ngt-group [position]="position()" [dispose]="null">
				<ngt-mesh castShadow [scale]="[0.61 * 6, 0.8 * 6, 6]" [geometry]="nodes.Cube.geometry">
					<ngts-mesh-transmission-material
						[options]="{
							backside: true,
							samples: 4,
							thickness: 3,
							anisotropy: 0.1,
							iridescence: 1,
							iridescenceIOR: 1,
							iridescenceThicknessRange: [0, 1400],
						}"
					/>
				</ngt-mesh>
				<ngt-group #group>
					<ng-content />
				</ngt-group>
			</ngt-group>
		}
	`,
	schemas: [CUSTOM_ELEMENTS_SCHEMA],
	changeDetection: ChangeDetectionStrategy.OnPush,
	imports: [NgtsMeshTransmissionMaterial],
})
export class Tank {
	position = input<NgtVector3>([0, 0, 0]);

	private groupRef = viewChild<ElementRef<Group>>('group');

	protected gltf = gltfResource<ShapesGLTF>(() => shapesGLB);

	private stencilParameters = mask(() => 1);

	constructor() {
		effect(() => {
			const gltf = this.gltf.value();
			if (!gltf) return;

			const group = this.groupRef()?.nativeElement;
			if (!group) return;

			const instanceState = getInstanceState(group);
			if (!instanceState) return;

			// track all children
			instanceState.objects();

			// Apply stencil to all contents
			group.traverse((child) => {
				if (child instanceof Mesh) {
					Object.assign(child.material, { ...this.stencilParameters() });
				}
			});
		});
	}
}
