import {
	ChangeDetectionStrategy,
	Component,
	CUSTOM_ELEMENTS_SCHEMA,
	effect,
	ElementRef,
	input,
	untracked,
	viewChild,
} from '@angular/core';
import { beforeRender, extend, injectStore, is, NgtThreeElements, pick } from 'angular-three';
import { mergeInputs } from 'ngxtension/inject-inputs';
import * as THREE from 'three';
import { Group } from 'three';

interface ControlsProto {
	update(): void;

	target: THREE.Vector3;
	maxDistance: number;
	addEventListener: (event: string, callback: (event: any) => void) => void;
	removeEventListener: (event: string, callback: (event: any) => void) => void;
}

enum AnimationState {
	NONE = 0,
	START = 1,
	ACTIVE = 2,
}

function interpolateFuncDefault(t: number) {
	// Imitates the previously used THREE.MathUtils.damp
	return 1 - Math.exp(-5 * t) + 0.007 * t;
}

/**
 * Configuration options for the NgtsBounds component.
 * Extends the standard ngt-group element options.
 */
export interface NgtsBoundsOptions extends Partial<NgtThreeElements['ngt-group']> {
	/**
	 * Maximum duration of the camera animation in seconds.
	 * @default 1.0
	 */
	maxDuration: number;
	/**
	 * Margin factor applied to the calculated camera distance.
	 * Values > 1 add padding around the bounds.
	 * @default 1.2
	 */
	margin: number;
	/**
	 * When enabled, automatically recalculates bounds when children change.
	 * @default false
	 */
	observe: boolean;
	/**
	 * When enabled, automatically fits the camera to the bounds on initialization.
	 * @default false
	 */
	fit: boolean;
	/**
	 * When enabled, automatically adjusts camera near/far planes to the bounds.
	 * @default false
	 */
	clip: boolean;
	/**
	 * Custom interpolation function for camera animation.
	 * Takes a value from 0 to 1 and returns an interpolated value.
	 * @default Damping-based interpolation function
	 */
	interpolateFunc: (t: number) => number;
}

const defaultOptions: NgtsBoundsOptions = {
	maxDuration: 1.0,
	margin: 1.2,
	interpolateFunc: interpolateFuncDefault,
	clip: false,
	fit: false,
	observe: false,
};

/**
 * A component that calculates the bounding box of its children and provides
 * methods to animate the camera to fit those bounds. Useful for focusing
 * the camera on specific objects or automatically framing content.
 *
 * Supports both perspective and orthographic cameras with appropriate
 * handling for each camera type.
 *
 * @example
 * ```html
 * <ngts-bounds [options]="{ fit: true, clip: true, observe: true }">
 *   <ngt-mesh>
 *     <ngt-box-geometry />
 *     <ngt-mesh-standard-material />
 *   </ngt-mesh>
 * </ngts-bounds>
 * ```
 */
@Component({
	selector: 'ngts-bounds',
	template: `
		<ngt-group #group>
			<ng-content />
		</ngt-group>
	`,
	schemas: [CUSTOM_ELEMENTS_SCHEMA],
	changeDetection: ChangeDetectionStrategy.OnPush,
})
export class NgtsBounds {
	options = input(defaultOptions, { transform: mergeInputs(defaultOptions) });

	groupRef = viewChild.required<ElementRef<THREE.Group>>('group');

	private store = injectStore();

	private clipOption = pick(this.options, 'clip');
	private fitOption = pick(this.options, 'fit');
	private observe = pick(this.options, 'observe');

	private origin = {
		camPos: new THREE.Vector3(),
		camRot: new THREE.Quaternion(),
		camZoom: 1,
	};
	private goal = {
		camPos: undefined as THREE.Vector3 | undefined,
		camRot: undefined as THREE.Quaternion | undefined,
		camZoom: undefined as number | undefined,
		camUp: undefined as THREE.Vector3 | undefined,
		target: undefined as THREE.Vector3 | undefined,
	};
	private animationState = AnimationState.NONE;

	// represent animation state from 0 to 1
	private t = 0;
	private box = new THREE.Box3();

	constructor() {
		extend({ Group });

		effect((onCleanup) => {
			const [controls, camera] = [this.store.controls() as unknown as ControlsProto, this.store.snapshot.camera];
			if (!controls) return;

			const callback = () => {
				if (this.goal.target && this.animationState !== AnimationState.NONE) {
					const front = new THREE.Vector3().setFromMatrixColumn(camera.matrix, 2);
					const d0 = this.origin.camPos.distanceTo(controls.target);
					const d1 = (this.goal.camPos || this.origin.camPos).distanceTo(this.goal.target);
					const d = (1 - this.t) * d0 + this.t * d1;

					controls.target.copy(camera.position).addScaledVector(front, -d);
					controls.update();
				}

				this.animationState = AnimationState.NONE;
			};

			controls.addEventListener('start', callback);
			onCleanup(() => controls.removeEventListener('start', callback));
		});

		let count = 0;
		effect(() => {
			const [clip, fit, observe] = [
				this.clipOption(),
				this.fitOption(),
				this.observe(),
				this.store.size(),
				this.store.camera(),
				this.store.controls(),
			];

			if (observe || count++ === 0) {
				this.refresh();
				if (fit) this.reset().fit();
				if (clip) this.clip();
			}
		});

		beforeRender(({ delta }) => {
			// This [additional animation step START] is needed to guarantee that delta used in animation isn't absurdly high (2-3 seconds) which is actually possible if rendering happens on demand...
			if (this.animationState === AnimationState.START) {
				this.animationState = AnimationState.ACTIVE;
				this.store.snapshot.invalidate();
			} else if (this.animationState === AnimationState.ACTIVE) {
				const [{ maxDuration, interpolateFunc }, camera, controls, invalidate] = [
					this.options(),
					this.store.snapshot.camera,
					this.store.snapshot.controls as unknown as ControlsProto,
					this.store.snapshot.invalidate,
				];
				this.t += delta / maxDuration;

				if (this.t >= 1) {
					this.goal.camPos && camera.position.copy(this.goal.camPos);
					this.goal.camRot && camera.quaternion.copy(this.goal.camRot);
					this.goal.camUp && camera.up.copy(this.goal.camUp);
					this.goal.camZoom && is.orthographicCamera(camera) && (camera.zoom = this.goal.camZoom);

					camera.updateMatrixWorld();
					camera.updateProjectionMatrix();

					if (controls && this.goal.target) {
						controls.target.copy(this.goal.target);
						controls.update();
					}

					this.animationState = AnimationState.NONE;
				} else {
					const k = interpolateFunc(this.t);

					this.goal.camPos && camera.position.lerpVectors(this.origin.camPos, this.goal.camPos, k);
					this.goal.camRot && camera.quaternion.slerpQuaternions(this.origin.camRot, this.goal.camRot, k);
					this.goal.camUp && camera.up.set(0, 1, 0).applyQuaternion(camera.quaternion);
					this.goal.camZoom &&
						is.orthographicCamera(camera) &&
						(camera.zoom = (1 - k) * this.origin.camZoom + k * this.goal.camZoom);

					camera.updateMatrixWorld();
					camera.updateProjectionMatrix();
				}

				invalidate();
			}
		});
	}

	/**
	 * Gets the current size information of the bounds.
	 *
	 * @returns An object containing the bounding box, size vector, center point, and calculated camera distance
	 */
	getSize() {
		const [camera, { margin }] = [this.store.snapshot.camera, untracked(this.options)];

		const boxSize = this.box.getSize(new THREE.Vector3());
		const center = this.box.getCenter(new THREE.Vector3());
		const maxSize = Math.max(boxSize.x, boxSize.y, boxSize.z);
		const fitHeightDistance = is.orthographicCamera(camera)
			? maxSize * 4
			: maxSize / (2 * Math.atan((Math.PI * camera.fov) / 360));
		const fitWidthDistance = is.orthographicCamera(camera) ? maxSize * 4 : fitHeightDistance / camera.aspect;
		const distance = margin * Math.max(fitHeightDistance, fitWidthDistance);

		return { box: this.box, size: boxSize, center, distance };
	}

	/**
	 * Refreshes the bounding box calculation from the given object or the component's children.
	 *
	 * @param object - Optional object or Box3 to calculate bounds from. If not provided, uses the component's group.
	 * @returns This instance for method chaining
	 */
	refresh(object?: THREE.Object3D | THREE.Box3) {
		const [group, camera] = [untracked(this.groupRef).nativeElement, this.store.snapshot.camera];

		if (is.three<THREE.Box3>(object, 'isBox3')) this.box.copy(object);
		else {
			const target = object || group;
			if (!target) return this;
			target.updateWorldMatrix(true, true);
			this.box.setFromObject(target);
		}
		if (this.box.isEmpty()) {
			const max = camera.position.length() || 10;
			this.box.setFromCenterAndSize(new THREE.Vector3(), new THREE.Vector3(max, max, max));
		}

		this.origin.camPos.copy(camera.position);
		this.origin.camRot.copy(camera.quaternion);
		is.orthographicCamera(camera) && (this.origin.camZoom = camera.zoom);

		this.goal.camPos = undefined;
		this.goal.camRot = undefined;
		this.goal.camZoom = undefined;
		this.goal.camUp = undefined;
		this.goal.target = undefined;

		return this;
	}

	/**
	 * Resets and animates the camera to fit the current bounds while maintaining
	 * the current viewing direction.
	 *
	 * @returns This instance for method chaining
	 */
	reset() {
		const [camera] = [this.store.snapshot.camera];
		const { center, distance } = this.getSize();

		const direction = camera.position.clone().sub(center).normalize();
		this.goal.camPos = center.clone().addScaledVector(direction, distance);
		this.goal.target = center.clone();
		const mCamRot = new THREE.Matrix4().lookAt(this.goal.camPos, this.goal.target, camera.up);
		this.goal.camRot = new THREE.Quaternion().setFromRotationMatrix(mCamRot);

		this.animationState = AnimationState.START;
		this.t = 0;

		return this;
	}

	/**
	 * Animates the camera to a new position.
	 *
	 * @param position - Target position as a Vector3 or [x, y, z] array
	 * @returns This instance for method chaining
	 */
	moveTo(position: THREE.Vector3 | [number, number, number]) {
		this.goal.camPos = Array.isArray(position) ? new THREE.Vector3(...position) : position.clone();

		this.animationState = AnimationState.START;
		this.t = 0;

		return this;
	}

	/**
	 * Animates the camera to look at a target position with optional up vector.
	 *
	 * @param params - Object containing target position and optional up vector
	 * @param params.target - The point to look at as a Vector3 or [x, y, z] array
	 * @param params.up - Optional up vector as a Vector3 or [x, y, z] array
	 * @returns This instance for method chaining
	 */
	lookAt({
		target,
		up,
	}: {
		target: THREE.Vector3 | [number, number, number];
		up?: THREE.Vector3 | [number, number, number];
	}) {
		const [camera] = [this.store.snapshot.camera];

		this.goal.target = Array.isArray(target) ? new THREE.Vector3(...target) : target.clone();
		if (up) {
			this.goal.camUp = Array.isArray(up) ? new THREE.Vector3(...up) : up.clone();
		} else {
			this.goal.camUp = camera.up.clone();
		}
		const mCamRot = new THREE.Matrix4().lookAt(
			this.goal.camPos || camera.position,
			this.goal.target,
			this.goal.camUp,
		);
		this.goal.camRot = new THREE.Quaternion().setFromRotationMatrix(mCamRot);

		this.animationState = AnimationState.START;
		this.t = 0;

		return this;
	}

	/**
	 * Fits the camera to the bounds by adjusting zoom (for orthographic cameras)
	 * or position (for perspective cameras).
	 *
	 * For orthographic cameras, this only adjusts the zoom level while preserving position.
	 * For perspective cameras, this behaves the same as reset().
	 *
	 * @returns This instance for method chaining
	 */
	fit() {
		const [camera, controls, { margin }] = [
			this.store.snapshot.camera,
			this.store.snapshot.controls as unknown as ControlsProto,
			untracked(this.options),
		];

		if (!is.orthographicCamera(camera)) {
			// For non-orthographic cameras, fit should behave exactly like reset
			return this.reset();
		}

		// For orthographic cameras, fit should only modify the zoom value
		let maxHeight = 0;
		let maxWidth = 0;
		const vertices = [
			new THREE.Vector3(this.box.min.x, this.box.min.y, this.box.min.z),
			new THREE.Vector3(this.box.min.x, this.box.max.y, this.box.min.z),
			new THREE.Vector3(this.box.min.x, this.box.min.y, this.box.max.z),
			new THREE.Vector3(this.box.min.x, this.box.max.y, this.box.max.z),
			new THREE.Vector3(this.box.max.x, this.box.max.y, this.box.max.z),
			new THREE.Vector3(this.box.max.x, this.box.max.y, this.box.min.z),
			new THREE.Vector3(this.box.max.x, this.box.min.y, this.box.max.z),
			new THREE.Vector3(this.box.max.x, this.box.min.y, this.box.min.z),
		];

		// Transform the center and each corner to camera space
		const pos = this.goal.camPos || camera.position;
		const target = this.goal.target || controls.target;
		const up = this.goal.camUp || camera.up;
		const mCamWInv = target
			? new THREE.Matrix4().lookAt(pos, target, up).setPosition(pos).invert()
			: camera.matrixWorldInverse;
		for (const v of vertices) {
			v.applyMatrix4(mCamWInv);
			maxHeight = Math.max(maxHeight, Math.abs(v.y));
			maxWidth = Math.max(maxWidth, Math.abs(v.x));
		}
		maxHeight *= 2;
		maxWidth *= 2;
		const zoomForHeight = (camera.top - camera.bottom) / maxHeight;
		const zoomForWidth = (camera.right - camera.left) / maxWidth;

		this.goal.camZoom = Math.min(zoomForHeight, zoomForWidth) / margin;

		this.animationState = AnimationState.START;
		this.t = 0;

		return this;
	}

	/**
	 * Adjusts the camera's near and far clipping planes based on the current bounds.
	 * Also updates the controls' maxDistance if controls are present.
	 *
	 * @returns This instance for method chaining
	 */
	clip() {
		const [camera, controls, invalidate] = [
			this.store.snapshot.camera,
			this.store.snapshot.controls as unknown as ControlsProto,
			this.store.snapshot.invalidate,
		];
		const { distance } = this.getSize();

		camera.near = distance / 100;
		camera.far = distance * 100;
		camera.updateProjectionMatrix();

		if (controls) {
			controls.maxDistance = distance * 10;
			controls.update();
		}

		invalidate();

		return this;
	}
}
