import { Subject } from 'rxjs';
import * as THREE from 'three';
import { getInstanceState } from './instance';
import type {
	NgtAnyRecord,
	NgtDomEvent,
	NgtEventHandlers,
	NgtIntersection,
	NgtPointerCaptureTarget,
	NgtState,
	NgtThreeEvent,
} from './types';
import { makeId } from './utils/make';
import { SignalState } from './utils/signal-state';

/**
 * @fileoverview Event handling system for Angular Three.
 *
 * This module provides the event handling infrastructure for raycasting-based
 * pointer events in Three.js scenes. It handles event propagation, pointer
 * capture, and event bubbling through the scene graph.
 */

/**
 * Releases pointer captures for an object.
 * Called by releasePointerCapture in the API, and when an object is removed.
 * @internal
 */
function releaseInternalPointerCapture(
	capturedMap: Map<number, Map<THREE.Object3D, NgtPointerCaptureTarget>>,
	obj: THREE.Object3D,
	captures: Map<THREE.Object3D, NgtPointerCaptureTarget>,
	pointerId: number,
): void {
	const captureData: NgtPointerCaptureTarget | undefined = captures.get(obj);
	if (captureData) {
		captures.delete(obj);
		// If this was the last capturing object for this pointer
		if (captures.size === 0) {
			capturedMap.delete(pointerId);
			captureData.target.releasePointerCapture(pointerId);
		}
	}
}

/**
 * Removes all traces of an object from the event handling system.
 *
 * This function cleans up:
 * - Interaction array
 * - Initial hits
 * - Hovered elements map
 * - Pointer captures
 *
 * @param store - The Angular Three store
 * @param object - The object to remove from interactivity
 */
export function removeInteractivity(store: SignalState<NgtState>, object: THREE.Object3D) {
	const { internal } = store.snapshot;
	// Removes every trace of an object from the data store
	internal.interaction = internal.interaction.filter((o) => o !== object);
	internal.initialHits = internal.initialHits.filter((o) => o !== object);
	internal.hovered.forEach((value, key) => {
		if (value.eventObject === object || value.object === object) {
			// Clear out intersects, they are outdated by now
			internal.hovered.delete(key);
		}
	});
	internal.capturedMap.forEach((captures, pointerId) => {
		releaseInternalPointerCapture(internal.capturedMap, object, captures, pointerId);
	});
}

/**
 * Creates the event handling system for a store.
 *
 * Returns an object with a `handlePointer` function that creates event handlers
 * for different pointer event types. These handlers perform raycasting,
 * event propagation, and callback invocation.
 *
 * @param store - The Angular Three store to create events for
 * @returns An object containing the handlePointer factory function
 */
export function createEvents(store: SignalState<NgtState>) {
	/** Calculates delta */
	function calculateDistance(event: NgtDomEvent) {
		const internal = store.snapshot.internal;
		const dx = event.offsetX - internal.initialClick[0];
		const dy = event.offsetY - internal.initialClick[1];
		return Math.round(Math.sqrt(dx * dx + dy * dy));
	}

	/** Returns true if an instance has a valid pointer-event registered, this excludes scroll, clicks etc */
	function filterPointerEvents(objects: THREE.Object3D[]) {
		return objects.filter((obj) =>
			['move', 'over', 'enter', 'out', 'leave'].some((name) => {
				const eventName = `pointer${name}` as keyof NgtEventHandlers;
				return getInstanceState(obj)?.handlers?.[eventName];
			}),
		);
	}

	function intersect(event: NgtDomEvent, filter?: (objects: THREE.Object3D[]) => THREE.Object3D[]) {
		const state = store.snapshot;
		const duplicates = new Set<string>();
		const intersections: NgtIntersection[] = [];
		// Allow callers to eliminate event objects
		const eventsObjects = filter ? filter(state.internal.interaction) : state.internal.interaction;

		if (!state.previousRoot) {
			// Make sure root-level pointer and ray are set up
			state.events.compute?.(event, store, null);
		}

		// Skip work if there are no event objects
		if (eventsObjects.length === 0) return intersections;

		// Reset all raycaster cameras to undefined - use for loop for better performance
		const eventsObjectsLen = eventsObjects.length;
		for (let i = 0; i < eventsObjectsLen; i++) {
			const objectRootState = getInstanceState(eventsObjects[i])?.store?.snapshot;
			if (objectRootState) {
				objectRootState.raycaster.camera = undefined!;
			}
		}

		// Pre-allocate array to avoid garbage collection
		const raycastResults: THREE.Intersection<THREE.Object3D>[] = [];

		function handleRaycast(obj: THREE.Object3D) {
			const objStore = getInstanceState(obj)?.store;
			const objState = objStore?.snapshot;
			// Skip event handling when noEvents is set, or when the raycasters camera is null
			if (!objState || !objState.events.enabled || objState.raycaster.camera === null) return [];

			// When the camera is undefined we have to call the event layers update function
			if (objState.raycaster.camera === undefined) {
				objState.events.compute?.(event, objStore, objState.previousRoot);
				// If the camera is still undefined we have to skip this layer entirely
				if (objState.raycaster.camera === undefined) objState.raycaster.camera = null!;
			}

			// Intersect object by object
			return objState.raycaster.camera ? objState.raycaster.intersectObject(obj, true) : [];
		}

		// Collect events
		for (let i = 0; i < eventsObjectsLen; i++) {
			const objResults = handleRaycast(eventsObjects[i]);
			if (objResults.length <= 0) continue;
			for (let j = 0; j < objResults.length; j++) {
				raycastResults.push(objResults[j]);
			}
		}

		// Sort by event priority and distance
		raycastResults.sort((a, b) => {
			const aState = getInstanceState(a.object)?.store?.snapshot;
			const bState = getInstanceState(b.object)?.store?.snapshot;
			if (!aState || !bState) return a.distance - b.distance;
			return bState.events.priority - aState.events.priority || a.distance - b.distance;
		});

		// Filter out duplicates - more efficient than chaining
		let hits: THREE.Intersection<THREE.Object3D>[] = [];
		for (let i = 0; i < raycastResults.length; i++) {
			const item = raycastResults[i];
			const id = makeId(item as NgtIntersection);
			if (duplicates.has(id)) continue;
			duplicates.add(id);
			hits.push(item);
		}

		// https://github.com/mrdoob/three.js/issues/16031
		// Allow custom userland intersect sort order, this likely only makes sense on the root filter
		if (state.events.filter) hits = state.events.filter(hits, store);

		// Bubble up the events, find the event source (eventObject)
		const hitsLen = hits.length;
		for (let i = 0; i < hitsLen; i++) {
			const hit = hits[i];
			let eventObject: THREE.Object3D | null = hit.object;
			// bubble event up
			while (eventObject) {
				if (getInstanceState(eventObject)?.eventCount) intersections.push({ ...hit, eventObject });
				eventObject = eventObject.parent;
			}
		}

		// If the interaction is captured, make all capturing targets part of the intersect.
		if ('pointerId' in event && state.internal.capturedMap.has(event.pointerId)) {
			const captures = state.internal.capturedMap.get(event.pointerId)!;
			for (const captureData of captures.values()) {
				if (duplicates.has(makeId(captureData.intersection))) continue;
				intersections.push(captureData.intersection);
			}
		}
		return intersections;
	}

	/**  Handles intersections by forwarding them to handlers */
	function handleIntersects(
		intersections: NgtIntersection[],
		event: NgtDomEvent,
		delta: number,
		callback: (event: NgtThreeEvent<NgtDomEvent>) => void,
	) {
		const rootState = store.snapshot;

		// If anything has been found, forward it to the event listeners
		if (intersections.length) {
			const localState = { stopped: false };
			for (const hit of intersections) {
				let instanceState = getInstanceState(hit.object);

				// If the object is not managed by NGT, it might be parented to an element which is.
				// Traverse upwards until we find a managed parent and use its state instead.
				if (!instanceState) {
					hit.object.traverseAncestors((ancestor) => {
						const parentInstanceState = getInstanceState(ancestor);
						if (parentInstanceState) {
							instanceState = parentInstanceState;
							return false;
						}
						return;
					});
				}

				const { raycaster, pointer, camera, internal } = instanceState?.store?.snapshot || rootState;

				const unprojectedPoint = new THREE.Vector3(pointer.x, pointer.y, 0).unproject(camera);
				const hasPointerCapture = (id: number) => internal.capturedMap.get(id)?.has(hit.eventObject) ?? false;

				const setPointerCapture = (id: number) => {
					const captureData = { intersection: hit, target: event.target as Element };
					if (internal.capturedMap.has(id)) {
						// if the pointerId was previously captured, we add the hit to the
						// event capturedMap.
						internal.capturedMap.get(id)!.set(hit.eventObject, captureData);
					} else {
						// if the pointerId was not previously captured, we create a map
						// containing the hitObject, and the hit. hitObject is used for
						// faster access.
						internal.capturedMap.set(id, new Map([[hit.eventObject, captureData]]));
					}
					// Call the original event now
					(event.target as Element).setPointerCapture(id);
				};

				const releasePointerCapture = (id: number) => {
					const captures = internal.capturedMap.get(id);
					if (captures) {
						releaseInternalPointerCapture(internal.capturedMap, hit.eventObject, captures, id);
					}
				};

				// Add native event props
				const extractEventProps: any = {};
				// This iterates over the event's properties including the inherited ones. Native PointerEvents have most of their props as getters which are inherited, but polyfilled PointerEvents have them all as their own properties (i.e. not inherited). We can't use Object.keys() or Object.entries() as they only return "own" properties; nor Object.getPrototypeOf(event) as that *doesn't* return "own" properties, only inherited ones.
				for (const prop in event) {
					const property = event[prop as keyof NgtDomEvent];
					// Only copy over atomics, leave functions alone as these should be
					// called as event.nativeEvent.fn()
					if (typeof property !== 'function') extractEventProps[prop] = property;
				}

				const raycastEvent: NgtThreeEvent<NgtDomEvent> = {
					...hit,
					...extractEventProps,
					pointer,
					intersections,
					stopped: localState.stopped,
					delta,
					unprojectedPoint,
					ray: raycaster.ray,
					camera,
					// Hijack stopPropagation, which just sets a flag
					stopPropagation() {
						// https://github.com/pmndrs/react-three-fiber/issues/596
						// Events are not allowed to stop propagation if the pointer has been captured
						const capturesForPointer = 'pointerId' in event && internal.capturedMap.get(event.pointerId);

						// We only authorize stopPropagation...
						if (
							// ...if this pointer hasn't been captured
							!capturesForPointer ||
							// ... or if the hit object is capturing the pointer
							capturesForPointer.has(hit.eventObject)
						) {
							raycastEvent.stopped = localState.stopped = true;
							// Propagation is stopped, remove all other hover records
							// An event handler is only allowed to flush other handlers if it is hovered itself
							if (
								internal.hovered.size &&
								Array.from(internal.hovered.values()).find((i) => i.eventObject === hit.eventObject)
							) {
								// Objects cannot flush out higher up objects that have already caught the event
								const higher = intersections.slice(0, intersections.indexOf(hit));
								cancelPointer([...higher, hit]);
							}
						}
					},
					// there should be a distinction between target and currentTarget
					target: { hasPointerCapture, setPointerCapture, releasePointerCapture },
					currentTarget: { hasPointerCapture, setPointerCapture, releasePointerCapture },
					nativeEvent: event,
				};

				// Call subscribers
				callback(raycastEvent);
				// Event bubbling may be interrupted by stopPropagation
				if (localState.stopped) break;
			}
		}
		return intersections;
	}

	function cancelPointer(intersections: NgtIntersection[]) {
		const internal = store.snapshot.internal;
		for (const hoveredObj of internal.hovered.values()) {
			// When no objects were hit or the the hovered object wasn't found underneath the cursor
			// we call onPointerOut and delete the object from the hovered-elements map
			if (
				!intersections.length ||
				!intersections.find(
					(hit) =>
						hit.object === hoveredObj.object &&
						hit.index === hoveredObj.index &&
						hit.instanceId === hoveredObj.instanceId,
				)
			) {
				const eventObject = hoveredObj.eventObject;
				const instance = getInstanceState(eventObject);
				const handlers = instance?.handlers;
				internal.hovered.delete(makeId(hoveredObj));
				if (instance?.eventCount) {
					// Clear out intersects, they are outdated by now
					const data = { ...hoveredObj, intersections };
					handlers?.pointerout?.(data as NgtThreeEvent<PointerEvent>);
					handlers?.pointerleave?.(data as NgtThreeEvent<PointerEvent>);
				}
			}
		}
	}

	function pointerMissed(event: MouseEvent, objects: THREE.Object3D[]) {
		for (let i = 0; i < objects.length; i++) {
			const instance = getInstanceState(objects[i]);
			instance?.handlers.pointermissed?.(event);
		}
	}

	function handlePointer(name: string) {
		// Handle common cancelation events
		if (name === 'pointerleave' || name === 'pointercancel') {
			return () => cancelPointer([]);
		}

		if (name === 'lostpointercapture') {
			return (event: NgtDomEvent) => {
				const { internal } = store.snapshot;
				if ('pointerId' in event && internal.capturedMap.has(event.pointerId)) {
					// If the object event interface had lostpointercapture, we'd call it here on every
					// object that's getting removed. We call it on the next frame because lostpointercapture
					// fires before pointerup. Otherwise pointerUp would never be called if the event didn't
					// happen in the object it originated from, leaving components in a in-between state.
					requestAnimationFrame(() => {
						// Only release if pointer-up didn't do it already
						if (internal.capturedMap.has(event.pointerId)) {
							internal.capturedMap.delete(event.pointerId);
							cancelPointer([]);
						}
					});
				}
			};
		}

		// Cache these values since they're used in the closure
		const isPointerMove = name === 'pointermove';
		const isClickEvent = name === 'click' || name === 'contextmenu' || name === 'dblclick';
		const filter = isPointerMove ? filterPointerEvents : undefined;

		// Any other pointer goes here ...
		return function handleEvent(event: NgtDomEvent) {
			// NOTE: __pointerMissed$ on NgtStore is private subject since we only expose the Observable
			const pointerMissed$: Subject<MouseEvent> = (store as NgtAnyRecord)['__pointerMissed$'];
			const internal = store.snapshot.internal;

			// Cache the event
			internal.lastEvent.nativeElement = event;

			// Get fresh intersects
			const hits = intersect(event, filter);
			// Only calculate distance for click events to avoid unnecessary math
			const delta = isClickEvent ? calculateDistance(event) : 0;

			// Save initial coordinates on pointer-down
			if (name === 'pointerdown') {
				internal.initialClick = [event.offsetX, event.offsetY];
				internal.initialHits = hits.map((hit) => hit.eventObject);
			}

			// Handle click miss events - early return optimization for better performance
			if (isClickEvent && hits.length === 0 && delta <= 2) {
				pointerMissed(event, internal.interaction);
				pointerMissed$.next(event);
				return; // Early return if nothing was hit
			}

			// Take care of unhover for pointer moves
			if (isPointerMove) cancelPointer(hits);

			// Define onIntersect handler - locally cache common properties for better performance
			function onIntersect(data: NgtThreeEvent<NgtDomEvent>) {
				const eventObject = data.eventObject;
				const instance = getInstanceState(eventObject);

				// Early return if no instance or event count
				if (!instance?.eventCount) return;

				const handlers = instance.handlers;
				if (!handlers) return;

				if (isPointerMove) {
					// Handle pointer move events
					const hasPointerOverHandlers = !!(
						handlers.pointerover ||
						handlers.pointerenter ||
						handlers.pointerout ||
						handlers.pointerleave
					);

					if (hasPointerOverHandlers) {
						const id = makeId(data);
						const hoveredItem = internal.hovered.get(id);
						if (!hoveredItem) {
							// If the object wasn't previously hovered, book it and call its handler
							internal.hovered.set(id, data);
							if (handlers.pointerover) handlers.pointerover(data as NgtThreeEvent<PointerEvent>);
							if (handlers.pointerenter) handlers.pointerenter(data as NgtThreeEvent<PointerEvent>);
						} else if (hoveredItem.stopped) {
							// If the object was previously hovered and stopped, we shouldn't allow other items to proceed
							data.stopPropagation();
						}
					}

					// Call mouse move
					if (handlers.pointermove) handlers.pointermove(data as NgtThreeEvent<PointerEvent>);
				} else {
					// All other events ...
					const handler = handlers[name as keyof NgtEventHandlers] as (
						event: NgtThreeEvent<PointerEvent>,
					) => void;

					if (handler) {
						// Forward all events back to their respective handlers with the exception of click events,
						// which must use the initial target
						if (!isClickEvent || internal.initialHits.includes(eventObject)) {
							// Get objects not in initialHits for pointer missed - avoid creating new arrays if possible
							const missedObjects = internal.interaction.filter(
								(object) => !internal.initialHits.includes(object),
							);

							// Call pointerMissed only if we have objects to notify
							if (missedObjects.length > 0) {
								pointerMissed(event, missedObjects);
							}

							// Now call the handler
							handler(data as NgtThreeEvent<PointerEvent>);
						}
					} else if (isClickEvent && internal.initialHits.includes(eventObject)) {
						// Trigger onPointerMissed on all elements that have pointer over/out handlers, but not click and weren't hit
						const missedObjects = internal.interaction.filter(
							(object) => !internal.initialHits.includes(object),
						);

						// Call pointerMissed only if we have objects to notify
						if (missedObjects.length > 0) {
							pointerMissed(event, missedObjects);
						}
					}
				}
			}

			// Process all intersections
			if (hits.length > 0) {
				handleIntersects(hits, event, delta, onIntersect);
			}
		};
	}

	return { handlePointer };
}
