import { ElementRef } from '@angular/core';
import { NgtThreeElement, resolveRef } from 'angular-three';
import * as THREE from 'three';

/**
 * Type definition for the PositionMesh element in Angular Three templates.
 */
export type NgtPositionMesh = NgtThreeElement<typeof PositionMesh>;

const _instanceLocalMatrix = new THREE.Matrix4();
const _instanceWorldMatrix = new THREE.Matrix4();
const _instanceIntersects: THREE.Intersection[] = [];
const _mesh = new THREE.Mesh<THREE.BufferGeometry, THREE.MeshBasicMaterial>();

/**
 * A virtual mesh class that represents a single instance within an InstancedMesh.
 *
 * PositionMesh extends THREE.Group and provides the ability to position, rotate,
 * scale, and color individual instances while maintaining proper raycasting support.
 * Each PositionMesh is linked to a parent InstancedMesh and contributes its transform
 * to the instance matrix buffer.
 *
 * This class enables individual instances to receive pointer events and have bounds
 * for frustum culling, which is not natively supported by THREE.InstancedMesh.
 */
export class PositionMesh extends THREE.Group {
	/**
	 * The color of this instance.
	 * @default new THREE.Color('white')
	 */
	color: THREE.Color;
	/**
	 * Reference to the parent InstancedMesh that this instance belongs to.
	 */
	instance: ElementRef<THREE.InstancedMesh> | THREE.InstancedMesh | null | undefined;

	constructor() {
		super();
		this.color = new THREE.Color('white');
		this.instance = undefined;
	}

	/**
	 * Gets the geometry from the parent InstancedMesh.
	 * This allows the virtual instance to have bounds for frustum culling.
	 */
	get geometry() {
		return resolveRef(this.instance)?.geometry;
	}

	/**
	 * Custom raycast implementation that enables this virtual instance to receive pointer events.
	 *
	 * @param raycaster - The raycaster to test against
	 * @param intersects - Array to populate with intersection results
	 */
	override raycast(raycaster: THREE.Raycaster, intersects: THREE.Intersection[]) {
		const parent = resolveRef(this.instance);
		if (!parent) return;
		if (!parent.geometry || !parent.material) return;
		_mesh.geometry = parent.geometry;
		const matrixWorld = parent.matrixWorld;
		const instanceId = parent.userData['instances'].indexOf(this);
		// If the instance wasn't found or exceeds the parents draw range, bail out
		if (instanceId === -1 || instanceId > parent.count) return;
		// calculate the world matrix for each instance
		parent.getMatrixAt(instanceId, _instanceLocalMatrix);
		_instanceWorldMatrix.multiplyMatrices(matrixWorld, _instanceLocalMatrix);
		// the mesh represents this single instance
		_mesh.matrixWorld = _instanceWorldMatrix;
		// raycast side according to instance material
		if (parent.material instanceof THREE.Material) _mesh.material.side = parent.material.side;
		else _mesh.material.side = parent.material[0].side;
		_mesh.raycast(raycaster, _instanceIntersects);
		// process the result of raycast
		for (let i = 0, l = _instanceIntersects.length; i < l; i++) {
			const intersect = _instanceIntersects[i];
			intersect.instanceId = instanceId;
			intersect.object = this;
			intersects.push(intersect);
		}
		_instanceIntersects.length = 0;
	}
}

declare global {
	interface HTMLElementTagNameMap {
		/**
		 * @extends ngt-group
		 * @rawOptions instance|color
		 */
		'ngt-position-mesh': NgtPositionMesh;
	}
}
