import {
	createEvents,
	NgtAnyRecord,
	NgtDomEvent,
	NgtEventManager,
	NgtEvents,
	NgtState,
	SignalState,
} from 'angular-three';

/**
 * @fileoverview DOM event handling for Angular Three canvas.
 *
 * This module provides the default pointer event handling system for
 * Angular Three, translating DOM pointer events to Three.js raycasting events.
 */

const DOM_EVENTS = {
	click: false,
	contextmenu: false,
	dblclick: false,
	wheel: false, // passive wheel errors with OrbitControls
	pointerdown: true,
	pointerup: true,
	pointerleave: true,
	pointermove: true,
	pointercancel: true,
	lostpointercapture: true,
} as const;

/**
 * List of supported pointer event names.
 */
export const supportedEvents = [
	'click',
	'contextmenu',
	'dblclick',
	'pointerup',
	'pointerdown',
	'pointerover',
	'pointerout',
	'pointerenter',
	'pointerleave',
	'pointermove',
	'pointermissed',
	'pointercancel',
	'wheel',
] as const;

/**
 * Creates the default pointer event manager for an Angular Three canvas.
 *
 * This function sets up event listeners on the canvas element and translates
 * DOM pointer events to Three.js raycasting events. It handles:
 * - Pointer position calculation
 * - Raycaster setup
 * - Event connection/disconnection
 * - Event propagation
 *
 * @param store - The Angular Three store
 * @returns An event manager for the canvas
 */
export function createPointerEvents(store: SignalState<NgtState>): NgtEventManager<HTMLElement> {
	const { handlePointer } = createEvents(store);

	return {
		priority: 1,
		enabled: true,
		compute: (event: NgtDomEvent, root: SignalState<NgtState>) => {
			const state = root.snapshot;
			// https://github.com/pmndrs/react-three-fiber/pull/782
			// Events trigger outside of canvas when moved, use offsetX/Y by default and allow overrides
			state.pointer.set((event.offsetX / state.size.width) * 2 - 1, -(event.offsetY / state.size.height) * 2 + 1);
			state.raycaster.setFromCamera(state.pointer, state.camera);
		},
		connected: undefined,
		handlers: Object.keys(DOM_EVENTS).reduce((handlers: NgtAnyRecord, supportedEventName) => {
			handlers[supportedEventName] = handlePointer(supportedEventName);
			return handlers;
		}, {}) as NgtEvents,
		update: () => {
			const { events, internal } = store.snapshot;
			if (internal.lastEvent?.nativeElement && events.handlers)
				events.handlers.pointermove(internal.lastEvent.nativeElement);
		},
		connect: (target: HTMLElement) => {
			const state = store.snapshot;
			state.events.disconnect?.();

			state.setEvents({ connected: target });

			Object.entries(state.events.handlers ?? {}).forEach(
				([eventName, eventHandler]: [string, EventListener]) => {
					const passive = DOM_EVENTS[eventName as keyof typeof DOM_EVENTS];
					target.addEventListener(eventName, eventHandler, { passive });
				},
			);
		},
		disconnect: () => {
			const { events, setEvents } = store.snapshot;
			if (events.connected) {
				Object.entries(events.handlers ?? {}).forEach(([eventName, eventHandler]: [string, EventListener]) => {
					if (events.connected instanceof HTMLElement) {
						events.connected.removeEventListener(eventName, eventHandler);
					}
				});

				setEvents({ connected: undefined });
			}
		},
	};
}
