import { computed, Directive, effect, inject, InjectionToken, input } from '@angular/core';
import { InteractionGroups } from '@dimforge/rapier3d-compat';

/**
 * Calculates an InteractionGroup bitmask for use in the `collisionGroups` or `solverGroups`
 * properties of RigidBody or Collider components. The first argument represents a list of
 * groups the entity is in (expressed as numbers from 0 to 15). The second argument is a list
 * of groups that will be filtered against. When it is omitted, all groups are filtered against.
 *
 * @example
 * A RigidBody that is member of group 0 and will collide with everything from groups 0 and 1:
 *
 * ```tsx
 * <RigidBody collisionGroups={interactionGroups([0], [0, 1])} />
 * ```
 *
 * A RigidBody that is member of groups 0 and 1 and will collide with everything else:
 *
 * ```tsx
 * <RigidBody collisionGroups={interactionGroups([0, 1])} />
 * ```
 *
 * A RigidBody that is member of groups 0 and 1 and will not collide with anything:
 *
 * ```tsx
 * <RigidBody collisionGroups={interactionGroups([0, 1], [])} />
 * ```
 *
 * Please note that Rapier needs interaction filters to evaluate to true between _both_ colliding
 * entities for collision events to trigger.
 *
 * @param memberships Groups the collider is a member of. (Values can range from 0 to 15.)
 * @param filters Groups the interaction group should filter against. (Values can range from 0 to 15.)
 * @returns An InteractionGroup bitmask.
 */
export function interactionGroups(memberships: number | number[], filters?: number | number[]): InteractionGroups {
	return (bitmask(memberships) << 16) + (filters !== undefined ? bitmask(filters) : 0b1111_1111_1111_1111);
}

function bitmask(groups: number | number[]): InteractionGroups {
	return [groups].flat().reduce((acc, layer) => acc | (1 << layer), 0);
}

/**
 * Injection token for handlers that set collision groups on colliders.
 * Used internally by rigid bodies and attractors to apply interaction groups.
 */
export const COLLISION_GROUPS_HANDLER = new InjectionToken<
	() => undefined | ((interactionGroups: InteractionGroups) => void)
>('COLLISION_GROUPS_HANDLER');

/**
 * Directive for setting collision/solver groups on a rigid body or collider.
 * This allows filtering which objects can collide with each other.
 *
 * @example
 * ```html
 * <!-- Object in group 0, collides with groups 0 and 1 -->
 * <ngt-object3D rigidBody [interactionGroups]="[[0], [0, 1]]">
 *   <ngt-mesh>
 *     <ngt-box-geometry />
 *   </ngt-mesh>
 * </ngt-object3D>
 *
 * <!-- Object in groups 0 and 1, collides with everything -->
 * <ngt-object3D rigidBody [interactionGroups]="[[0, 1]]">
 *   <ngt-mesh>
 *     <ngt-sphere-geometry />
 *   </ngt-mesh>
 * </ngt-object3D>
 * ```
 */
@Directive({ selector: 'ngt-object3D[interactionGroups]' })
export class NgtrInteractionGroups {
	inputs = input.required<[number | number[], (number | number[])?]>({ alias: 'interactionGroups' });
	interactionGroups = computed(() => {
		const [memberships, filters] = this.inputs();
		return interactionGroups(memberships, filters);
	});

	constructor() {
		const collisionGroupsHandlerFn = inject(COLLISION_GROUPS_HANDLER, { host: true, optional: true });

		effect(() => {
			if (!collisionGroupsHandlerFn) return;
			const handler = collisionGroupsHandlerFn();
			if (!handler) return;
			handler(this.interactionGroups());
		});
	}
}
