import {
	DestroyRef,
	Directive,
	EmbeddedViewRef,
	Injector,
	ResourceRef,
	Signal,
	TemplateRef,
	ViewContainerRef,
	computed,
	effect,
	inject,
	input,
	signal,
} from '@angular/core';
import { injectTexture, textureResource } from 'angular-three-soba/loaders';
import { assertInjector } from 'ngxtension/assert-injector';
import * as THREE from 'three';

function getFormatString(format: number) {
	switch (format) {
		case 64:
			return '-64px';
		case 128:
			return '-128px';
		case 256:
			return '-256px';
		case 512:
			return '-512px';
		default:
			return '';
	}
}

const LIST_URL = 'https://cdn.jsdelivr.net/gh/pmndrs/drei-assets@master/matcaps.json';
const MATCAP_ROOT = 'https://rawcdn.githack.com/emmelleppi/matcaps/9b36ccaaf0a24881a39062d05566c9e92be4aa0d';

/**
 * Injects a matcap texture from the matcaps library.
 * Matcaps provide realistic lighting without actual lights in the scene.
 *
 * @deprecated Use `matcapTextureResource` instead. Will be removed in v5.0.0.
 * @since v4.0.0
 *
 * @param id - Signal of matcap ID (number index or string hash)
 * @param config - Configuration options
 * @param config.format - Signal of texture resolution (64, 128, 256, 512, or 1024)
 * @param config.onLoad - Callback when texture is loaded
 * @param config.injector - Optional injector for dependency injection
 * @returns Object with url, texture signal, and numTot (total available matcaps)
 *
 * @example
 * ```typescript
 * const { texture } = injectMatcapTexture(() => 42);
 * ```
 */
export function injectMatcapTexture(
	id: () => number | string = () => 0,
	{
		format = () => 1024,
		onLoad,
		injector,
	}: { format?: () => number; onLoad?: (texture: THREE.Texture[]) => void; injector?: Injector } = {},
) {
	return assertInjector(injectMatcapTexture, injector, () => {
		const matcapList = signal<Record<string, string>>({});

		fetch(LIST_URL)
			.then((res) => res.json())
			.then((list) => {
				matcapList.set(list);
			});

		const DEFAULT_MATCAP = computed(() => matcapList()[0]);
		const numTot = computed(() => Object.keys(matcapList()).length);

		const fileHash = computed(() => {
			const idValue = id();
			if (typeof idValue === 'string') {
				return idValue;
			}

			if (typeof idValue === 'number') {
				return matcapList()[idValue];
			}

			return null;
		});

		const fileName = computed(() => `${fileHash() || DEFAULT_MATCAP()}${getFormatString(format())}.png`);
		const url = computed(() => `${MATCAP_ROOT}/${format()}/${fileName()}`);

		const matcapTexture = injectTexture(url, { onLoad });

		return { url, texture: matcapTexture, numTot };
	});
}

/**
 * Creates a reactive resource for loading matcap textures.
 * Matcaps provide realistic lighting without actual lights in the scene.
 *
 * @param id - Signal of matcap ID (number index or string hash)
 * @param config - Configuration options
 * @param config.format - Signal of texture resolution (64, 128, 256, 512, or 1024)
 * @param config.onLoad - Callback when texture is loaded
 * @param config.injector - Optional injector for dependency injection
 * @returns Object with url, resource, and numTot (total available matcaps)
 *
 * @example
 * ```typescript
 * const { resource } = matcapTextureResource(() => 42);
 * // Access texture: resource.value()
 * ```
 */
export function matcapTextureResource(
	id: () => number | string = () => 0,
	{
		format = () => 1024,
		onLoad,
		injector,
	}: { format?: () => number; onLoad?: (texture: THREE.Texture) => void; injector?: Injector } = {},
) {
	return assertInjector(matcapTextureResource, injector, () => {
		const matcapList = signal<Record<string, string>>({});

		fetch(LIST_URL)
			.then((res) => res.json())
			.then((list) => {
				matcapList.set(list);
			});

		const DEFAULT_MATCAP = computed(() => matcapList()[0]);
		const numTot = computed(() => Object.keys(matcapList()).length);

		const fileHash = computed(() => {
			const idValue = id();
			if (typeof idValue === 'string') {
				return idValue;
			}

			if (typeof idValue === 'number') {
				return matcapList()[idValue];
			}

			return null;
		});

		const fileName = computed(() => `${fileHash() || DEFAULT_MATCAP()}${getFormatString(format())}.png`);
		const url = computed(() => `${MATCAP_ROOT}/${format()}/${fileName()}`);

		const resource = textureResource(url, { onLoad });
		return { url, resource, numTot };
	});
}

/**
 * Options for the NgtsMatcapTexture structural directive.
 */
export interface NgtsMatcapTextureOptions {
	/**
	 * Matcap ID (number index or string hash).
	 * @default 0
	 */
	id?: number | string;
	/**
	 * Texture resolution (64, 128, 256, 512, or 1024).
	 * @default 1024
	 */
	format?: number;
}

/**
 * Structural directive for loading and using matcap textures in templates.
 * Provides the loaded texture resource through the template context.
 *
 * @example
 * ```html
 * <ng-template [matcapTexture]="{ id: 42, format: 512 }" let-resource>
 *   @if (resource.hasValue()) {
 *     <ngt-mesh-matcap-material [matcap]="resource.value()" />
 *   }
 * </ng-template>
 * ```
 */
@Directive({ selector: 'ng-template[matcapTexture]' })
export class NgtsMatcapTexture {
	matcapTexture = input<NgtsMatcapTextureOptions>();
	matcapTextureLoaded = input<(texture: THREE.Texture) => void>();

	private template = inject(TemplateRef);
	private vcr = inject(ViewContainerRef);

	private id = computed(() => this.matcapTexture()?.id ?? 0);
	private format = computed(() => this.matcapTexture()?.format ?? 1024);

	private ref?: EmbeddedViewRef<{ $implicit: Signal<THREE.Texture | null> }>;

	constructor() {
		const { resource } = matcapTextureResource(this.id, {
			format: this.format,
			onLoad: this.matcapTextureLoaded(),
		});

		effect(() => {
			this.ref = this.vcr.createEmbeddedView(this.template, { $implicit: resource });
			this.ref.detectChanges();
		});

		inject(DestroyRef).onDestroy(() => {
			this.ref?.destroy();
		});
	}

	static ngTemplateContextGuard(
		_: NgtsMatcapTexture,
		ctx: unknown,
	): ctx is { $implicit: ResourceRef<THREE.Texture | undefined> } {
		return true;
	}
}
