const std = @import("std");
const Build = std.Build;
const Step = Build.Step;
const LazyPath = Build.LazyPath;
const Compile = Step.Compile;
const Run = Step.Run;
const UpdateSourceFiles = Step.UpdateSourceFiles;
const WriteFile = Step.WriteFile;
const ResolvedTarget = Build.ResolvedTarget;
const OptimizeMode = std.builtin.OptimizeMode;

// import build tools from r_build_zig
const r_build_zig = @import("r_build_zig");

// import generated build.zig
const generated_build = @import("build-aux/generated/build.zig");

pub fn build(b: *std.Build) !void {
    // const buf: [std.mem.page_size]u8 = undefined;
    const target = b.standardTargetOptions(.{});
    const config_path = "build-aux/config.json";

    const assets = b.option(
        []const u8,
        "assets",
        "Path to directory with R dependencies for offline build",
    );

    // additional steps
    const update_step = b.step("update", "Generate R package build files");
    const tarball_step = b.step("dist", "Make a source archive");
    const fat_tarball_step = b.step("dist-fat", "Make a source archive with all R dependencies");

    // step dist-fat requires assets option
    if (assets == null)
        fat_tarball_step.dependOn(&b.addFail("The dist-fat step requires -Dassets to be set, e.g -Dassets=zig-out/assets.").step);

    // check if zig/cache/p exists, otherwise give dist-fat error
    std.fs.cwd().access("zig/cache/p", .{}) catch |e| switch (e) {
        error.FileNotFound => fat_tarball_step.dependOn(&b.addFail("The dist-fat step requires a zig cache in zig/cache. Before running dist-fat, build with `zig build --global-cache-dir zig/cache`").step),
        else => return e,
    };

    // declare build install rules
    try fetch_assets_and_build(b, config_path, target, .ReleaseSafe, assets);

    // declare rules for conf
    build_conf(b);

    // declare step: update
    try generate_build_script(
        b,
        config_path,
        &.{ // dirs relative to this build.zig file, and no trailing slashes
            "packages",
            "rcloud.client",
            "rcloud.packages",
            "rcloud.support",
        },
        update_step,
        target,
        .ReleaseSafe,
    );

    // declare step: dist
    try make_tarball(b, tarball_step);

    // declare step: dist-fat
    try make_fat_tarball(b, fat_tarball_step, assets);
}

/// Declare steps which fetch external assets and declare the rules
/// generated by the generate-build tool.
fn fetch_assets_and_build(
    b: *Build,
    config_path: []const u8,
    target: ResolvedTarget,
    optimize: OptimizeMode,
    assets: ?[]const u8,
) !void {
    if (assets) |assets_dir| {
        // we are doing an offline build
        const assets_path = b.path(assets_dir);

        // supply assets directory to build rule declarations
        try generated_build.build(b, assets_path);

        // add extra rcloud targets.
        // TODO: get rid of this when we move rcloud.solr into this repository
        add_extra_rcloud_targets(b, assets_path);

        // declare rules for offline htdocs, which is just a copy of source files
        build_htdocs_offline(b);
    } else {
        // we are doing a standard online build

        // get the fetch-assets tool
        const exe = b.dependency("r_build_zig", .{
            .target = target,
            .optimize = optimize,
        }).artifact("fetch-assets");

        // run it
        const step = b.addRunArtifact(exe);
        _ = step.addFileArg(b.path(config_path));
        const out_dir = step.addOutputDirectoryArg("assets");

        // add install step to copy assets directory to prefix
        const assets_install = b.addInstallDirectory(.{
            .source_dir = out_dir,
            .install_dir = .prefix,
            .install_subdir = "assets",
        });

        // install assets
        b.getInstallStep().dependOn(&assets_install.step);

        // supply output directory to build rule declarations
        try generated_build.build(b, out_dir);

        // add extra rcloud targets.
        // TODO: get rid of this when we move rcloud.solr into this repository
        add_extra_rcloud_targets(b, out_dir);

        // declare rules for htdocs, which require an online build
        build_htdocs(b);
    }
}

/// Declare steps which generate a new build script given a
/// configuration file and list of source package directories relative
/// to this script (in the project root).
fn generate_build_script(
    b: *Build,
    config_path: []const u8,
    relative_source_package_paths: []const []const u8,
    update_step: *Step,
    target: ResolvedTarget,
    optimize: OptimizeMode,
) !void {
    const exe = b.dependency("r_build_zig", .{
        .target = target,
        .optimize = optimize,
    }).artifact("generate-build");

    // arguments: config_file out_dir package_dirs...
    const step = b.addRunArtifact(exe);
    step.has_side_effects = true;

    // the tool wants a relative path, so we use addArg, and manually
    // add the dependency with addFileInput
    _ = step.addArg(config_path);
    _ = step.addFileInput(b.path(config_path));

    const out_dir = step.addOutputDirectoryArg("deps");
    for (relative_source_package_paths) |path| {
        _ = step.addArg(path);
    }

    // copy the generated build.zig file to build-aux directory
    const uf = b.addUpdateSourceFiles();
    uf.addCopyFileToSource(out_dir.path(b, "build.zig"), "build-aux/generated/build.zig");

    update_step.dependOn(&uf.step);
}

fn make_tarball(
    b: *Build,
    step: *Step,
) !void {
    const version = try read_version_file(b.allocator);
    const dirname = b.fmt("rcloud-{s}", .{version});
    const tarname = b.fmt("rcloud-{s}.tar.gz", .{version});

    const wf = b.addWriteFiles();
    const tar = b.addSystemCommand(&.{ "tar", "czf" });
    tar.setCwd(wf.getDirectory());
    const tar_out = tar.addOutputFileArg(tarname);
    _ = tar.addArg(dirname);
    add_all_source_files(b, wf, dirname);

    const tar_install = b.addInstallFileWithDir(tar_out, .prefix, tarname);
    step.dependOn(&tar_install.step);
}

fn make_fat_tarball(
    b: *Build,
    step: *Step,
    assets: ?[]const u8,
) !void {
    if (assets == null) {
        return;
    }

    const version = try read_version_file(b.allocator);
    const dirname = b.fmt("rcloud-full-{s}", .{version});
    const tarname = b.fmt("rcloud-full-{s}.tar.gz", .{version});

    const wf = b.addWriteFiles();
    const tar = b.addSystemCommand(&.{ "tar", "czf" });
    tar.setCwd(wf.getDirectory());
    const tar_out = tar.addOutputFileArg(tarname);
    _ = tar.addArg(dirname);
    add_all_source_files_and_assets(b, wf, dirname, assets.?);

    const tar_install = b.addInstallFileWithDir(tar_out, .prefix, tarname);
    step.dependOn(&tar_install.step);
}

fn add_all_source_files(b: *Build, wf: *WriteFile, dirname: []const u8) void {
    const options = WriteFile.Directory.Options{
        .exclude_extensions = &.{},
    };

    _ = add_copy_directory(b, wf, "conf", dirname, options);
    _ = add_copy_directory(b, wf, "doc", dirname, options);
    _ = add_copy_directory(b, wf, "docker", dirname, options);
    _ = add_copy_directory(b, wf, "htdocs", dirname, options);
    _ = add_copy_directory(b, wf, "packages", dirname, options);
    _ = add_copy_directory(b, wf, "rcloud.client", dirname, options);
    _ = add_copy_directory(b, wf, "rcloud.packages", dirname, options);
    _ = add_copy_directory(b, wf, "rcloud.support", dirname, options);
    _ = add_copy_directory(b, wf, "scripts", dirname, options);
    _ = add_copy_directory(b, wf, "services", dirname, options);
    _ = add_copy_directory(b, wf, "packages", dirname, options);

    _ = add_copy_file(b, wf, "build-aux/config.json", dirname);
    _ = add_copy_file(b, wf, "build-aux/generated/build.zig", dirname);

    _ = add_copy_file(b, wf, "zig/download.sh", dirname);
    _ = add_copy_file(b, wf, "zig/download.sh-README", dirname);
    _ = add_copy_file(b, wf, "zig/download.sh-LICENSE", dirname);

    _ = add_copy_file(b, wf, "build.zig", dirname);
    _ = add_copy_file(b, wf, "build.zig.zon", dirname);
    _ = add_copy_file(b, wf, "flake.lock", dirname);
    _ = add_copy_file(b, wf, "flake.nix", dirname);
    _ = add_copy_file(b, wf, "Gruntfile.js", dirname);
    _ = add_copy_file(b, wf, "LICENSE", dirname);
    _ = add_copy_file(b, wf, "NEWS.md", dirname);
    _ = add_copy_file(b, wf, "package.json", dirname);
    _ = add_copy_file(b, wf, "package-lock.json", dirname);
    _ = add_copy_file(b, wf, "README.md", dirname);
    _ = add_copy_file(b, wf, "VERSION", dirname);
}

fn add_all_source_files_and_assets(b: *Build, wf: *WriteFile, dirname: []const u8, assets: []const u8) void {
    const options = WriteFile.Directory.Options{
        .exclude_extensions = &.{},
    };

    add_all_source_files(b, wf, dirname);

    // include assets directory
    _ = add_copy_directory(b, wf, assets, dirname, options);

    // add zig cache directory which includes the zig build dependency
    // source files (not binaries). TODO: this path depends on the
    // maintainer having properly built rcloud using the command `zig
    // build --global-cache-dir zig/cache`
    _ = add_copy_directory(b, wf, "zig/cache/p", dirname, options);

    // include generated javascript bundles. TODO: these paths depend
    // on the default install prefix, zig-out. It will fail otherwise.
    _ = add_copy_file(b, wf, "zig-out/htdocs/css/rcloud.css", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/css/rcloud.css.map", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/css/rcloud-discover.css", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/css/rcloud-discover.css.map", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/css/rcloud-edit.css", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/css/rcloud-edit.css.map", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/css/rcloud-view.css", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/css/rcloud-view.css.map", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/css/rcloud-viewer.css", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/css/rcloud-viewer.css.map", dirname);

    _ = add_copy_file(b, wf, "zig-out/htdocs/js/merger_bundle.js", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/js/merger_bundle.js.gz", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/js/merger_bundle.js.map", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/js/merger_bundle.min.js", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/js/merger_bundle.min.js.gz", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/js/merger_bundle.min.js.map", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/js/rcloud_bundle.js", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/js/rcloud_bundle.js.gz", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/js/rcloud_bundle.js.map", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/js/rcloud_bundle.min.js", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/js/rcloud_bundle.min.js.gz", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/js/rcloud_bundle.min.js.map", dirname);

    _ = add_copy_file(b, wf, "zig-out/htdocs/lib/ace_bundle.js.gz", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/lib/ace_bundle.min.js", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/lib/ace_bundle.min.js.gz", dirname);
    _ = add_copy_file(b, wf, "zig-out/htdocs/lib/ace_bundle.min.js.map", dirname);
}

fn add_copy_directory(
    b: *Build,
    wf: *WriteFile,
    name: []const u8,
    root: []const u8,
    options: WriteFile.Directory.Options,
) void {
    _ = wf.addCopyDirectory(b.path(name), b.fmt("{s}/{s}", .{ root, name }), options);
}

fn add_copy_file(b: *Build, wf: *WriteFile, name: []const u8, root: []const u8) void {
    _ = wf.addCopyFile(b.path(name), b.fmt("{s}/{s}", .{ root, name }));
}

fn read_version_file(alloc: std.mem.Allocator) ![]const u8 {
    const file = try std.fs.cwd().openFile("VERSION", .{});
    defer file.close();
    const content = try file.readToEndAlloc(alloc, 1024);
    var i: usize = 0;
    while (i < content.len) : (i += 1) {
        if (content[i] == '\n') return content[0..i];
    }
    return content;
}

fn build_htdocs(b: *Build) void {
    const wf = b.addWriteFiles();

    // copy htdocs source
    _ = wf.addCopyDirectory(b.path("htdocs"), "htdocs", .{});

    // install js requirements
    const npm_ci = b.addSystemCommand(&.{ "npm", "ci" });
    npm_ci.setCwd(wf.getDirectory());
    npm_ci.addFileInput(wf.addCopyFile(b.path("package.json"), "package.json"));
    npm_ci.addFileInput(wf.addCopyFile(b.path("package-lock.json"), "package-lock.json"));
    npm_ci.expectExitCode(0);

    // run grunt
    const grunt = b.addSystemCommand(&.{"node_modules/grunt-cli/bin/grunt"});
    grunt.setCwd(wf.getDirectory());
    grunt.addFileInput(wf.addCopyFile(b.path("Gruntfile.js"), "Gruntfile.js"));
    grunt.addFileInput(wf.addCopyFile(b.path("VERSION"), "VERSION"));
    grunt.addFileInput(wf.getDirectory().path(b, "node_modules/grunt-cli/bin/grunt"));
    grunt.expectExitCode(0);

    // which depends on npm_ci
    grunt.step.dependOn(&npm_ci.step);

    // add an install step for post-grunt htdocs
    const htdocs_install = b.addInstallDirectory(.{
        .source_dir = wf.getDirectory().path(b, "htdocs"),
        .install_dir = .prefix,
        .install_subdir = "htdocs",
    });
    htdocs_install.step.dependOn(&grunt.step);

    // install built htdocs files
    b.getInstallStep().dependOn(&htdocs_install.step);
}

/// An offline build of htdocs, just a copy.
fn build_htdocs_offline(b: *Build) void {
    const wf = b.addWriteFiles();

    // copy htdocs source
    _ = wf.addCopyDirectory(b.path("htdocs"), "htdocs", .{});

    // add an install step for post-grunt htdocs
    const htdocs_install = b.addInstallDirectory(.{
        .source_dir = wf.getDirectory().path(b, "htdocs"),
        .install_dir = .prefix,
        .install_subdir = "htdocs",
    });

    // install built htdocs files
    b.getInstallStep().dependOn(&htdocs_install.step);
}

fn build_conf(b: *Build) void {
    // install conf directory
    b.getInstallStep().dependOn(&b.addInstallDirectory(.{
        .source_dir = b.path("conf"),
        .install_dir = .prefix,
        .install_subdir = "conf",
    }).step);

    // install VERSION file
    b.getInstallStep().dependOn(&b.addInstallFile(
        b.path("VERSION"),
        "VERSION",
    ).step);
}

fn add_extra_rcloud_targets(b: *Build, asset_dir: LazyPath) void {
    // FIXME: This is a hack until rcloud.solr is added to this repository's source directories.
    const libdir = b.addWriteFiles();

    const ulog = b.addSystemCommand(&.{"R"});
    ulog.addArgs(&.{
        "CMD",
        "INSTALL",
        "--no-docs",
        "--no-multiarch",
        "-l",
    });
    _ = ulog.addDirectoryArg(libdir.getDirectory());
    _ = ulog.addFileArg(asset_dir.path(b, "ulog_0.1-2.tar.gz"));
    ulog.step.name = "ulog";
    const ulog_out = ulog.captureStdOut();
    _ = ulog.captureStdErr();
    const ulog_install = b.addInstallDirectory(.{
        .source_dir = libdir.getDirectory().path(b, "ulog"),
        .install_dir = .{ .custom = "lib" },
        .install_subdir = "ulog",
    });

    ulog_install.step.dependOn(&ulog.step);
    b.getInstallStep().dependOn(&ulog_install.step);
    b.getInstallStep().dependOn(&b.addInstallFileWithDir(ulog_out, .{ .custom = "logs" }, "ulog.log").step);

    //

    const rcloud_solr = b.addSystemCommand(&.{"R"});
    rcloud_solr.addArgs(&.{
        "CMD",
        "INSTALL",
        "--no-docs",
        "--no-multiarch",
        "-l",
    });
    _ = rcloud_solr.addDirectoryArg(libdir.getDirectory());
    _ = rcloud_solr.addFileArg(asset_dir.path(b, "rcloud.solr_0.3.8.tar.gz"));
    rcloud_solr.step.name = "rcloud.solr";
    const rcloud_solr_out = rcloud_solr.captureStdOut();
    _ = rcloud_solr.captureStdErr();

    const rcloud_solr_install = b.addInstallDirectory(.{
        .source_dir = libdir.getDirectory().path(b, "rcloud.solr"),
        .install_dir = .{ .custom = "lib" },
        .install_subdir = "rcloud.solr",
    });

    rcloud_solr_install.step.dependOn(&rcloud_solr.step);
    b.getInstallStep().dependOn(&rcloud_solr_install.step);
    b.getInstallStep().dependOn(&b.addInstallFileWithDir(rcloud_solr_out, .{ .custom = "logs" }, "rcloud.solr.log").step);

    // this is so unfortunate but needed until we can get rid of this hack.
    for (b.getInstallStep().dependencies.items) |install_step| {
        for (install_step.dependencies.items) |step| {
            if (std.mem.eql(u8, "rcloud.support", step.name)) {
                rcloud_solr.step.dependOn(step);
            } else if (std.mem.eql(u8, "Rserve", step.name)) {
                rcloud_solr.step.dependOn(step);
            } else if (std.mem.eql(u8, "httr", step.name)) {
                rcloud_solr.step.dependOn(step);
            } else if (std.mem.eql(u8, "ulog", step.name)) {
                rcloud_solr.step.dependOn(step);
            } else if (std.mem.eql(u8, "R6", step.name)) {
                rcloud_solr.step.dependOn(step);
            }
        }
    }
}
