package agentssh

import (
	"context"
	"os"
	"path/filepath"
	"testing"

	"github.com/google/go-cmp/cmp"
	"github.com/spf13/afero"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func Test_addXauthEntry(t *testing.T) {
	t.Parallel()

	type testEntry struct {
		address      string
		display      string
		authProtocol string
		authCookie   string
	}
	tests := []struct {
		name         string
		authFile     []byte
		wantAuthFile []byte
		entries      []testEntry
	}{
		{
			name:     "add entry",
			authFile: nil,
			wantAuthFile: []byte{
				// w/unix:0  MIT-MAGIC-COOKIE-1  00
				//
				// 00000000: 0100 0001 7700 0130 0012 4d49 542d 4d41  ....w..0..MIT-MA
				// 00000010: 4749 432d 434f 4f4b 4945 2d31 0001 00    GIC-COOKIE-1...
				0x01, 0x00, 0x00, 0x01, 0x77, 0x00, 0x01, 0x30,
				0x00, 0x12, 0x4d, 0x49, 0x54, 0x2d, 0x4d, 0x41,
				0x47, 0x49, 0x43, 0x2d, 0x43, 0x4f, 0x4f, 0x4b,
				0x49, 0x45, 0x2d, 0x31, 0x00, 0x01, 0x00,
			},
			entries: []testEntry{
				{
					address:      "w",
					display:      "0",
					authProtocol: "MIT-MAGIC-COOKIE-1",
					authCookie:   "00",
				},
			},
		},
		{
			name:     "add two entries",
			authFile: []byte{},
			wantAuthFile: []byte{
				// w/unix:0  MIT-MAGIC-COOKIE-1  00
				// w/unix:1  MIT-MAGIC-COOKIE-1  11
				//
				// 00000000: 0100 0001 7700 0130 0012 4d49 542d 4d41  ....w..0..MIT-MA
				// 00000010: 4749 432d 434f 4f4b 4945 2d31 0001 0001  GIC-COOKIE-1....
				// 00000020: 0000 0177 0001 3100 124d 4954 2d4d 4147  ...w..1..MIT-MAG
				// 00000030: 4943 2d43 4f4f 4b49 452d 3100 0111       IC-COOKIE-1...
				0x01, 0x00, 0x00, 0x01, 0x77, 0x00, 0x01, 0x30,
				0x00, 0x12, 0x4d, 0x49, 0x54, 0x2d, 0x4d, 0x41,
				0x47, 0x49, 0x43, 0x2d, 0x43, 0x4f, 0x4f, 0x4b,
				0x49, 0x45, 0x2d, 0x31, 0x00, 0x01, 0x00,
				0x01, 0x00, 0x00, 0x01, 0x77, 0x00, 0x01, 0x31,
				0x00, 0x12, 0x4d, 0x49, 0x54, 0x2d, 0x4d, 0x41,
				0x47, 0x49, 0x43, 0x2d, 0x43, 0x4f, 0x4f, 0x4b,
				0x49, 0x45, 0x2d, 0x31, 0x00, 0x01, 0x11,
			},
			entries: []testEntry{
				{
					address:      "w",
					display:      "0",
					authProtocol: "MIT-MAGIC-COOKIE-1",
					authCookie:   "00",
				},
				{
					address:      "w",
					display:      "1",
					authProtocol: "MIT-MAGIC-COOKIE-1",
					authCookie:   "11",
				},
			},
		},
		{
			name: "update entry with new auth cookie length",
			authFile: []byte{
				// w/unix:0  MIT-MAGIC-COOKIE-1  00
				// w/unix:1  MIT-MAGIC-COOKIE-1  11
				//
				// 00000000: 0100 0001 7700 0130 0012 4d49 542d 4d41  ....w..0..MIT-MA
				// 00000010: 4749 432d 434f 4f4b 4945 2d31 0001 0001  GIC-COOKIE-1....
				// 00000020: 0000 0177 0001 3100 124d 4954 2d4d 4147  ...w..1..MIT-MAG
				// 00000030: 4943 2d43 4f4f 4b49 452d 3100 0111       IC-COOKIE-1...
				0x01, 0x00, 0x00, 0x01, 0x77, 0x00, 0x01, 0x30,
				0x00, 0x12, 0x4d, 0x49, 0x54, 0x2d, 0x4d, 0x41,
				0x47, 0x49, 0x43, 0x2d, 0x43, 0x4f, 0x4f, 0x4b,
				0x49, 0x45, 0x2d, 0x31, 0x00, 0x01, 0x00,
				0x01, 0x00, 0x00, 0x01, 0x77, 0x00, 0x01, 0x31,
				0x00, 0x12, 0x4d, 0x49, 0x54, 0x2d, 0x4d, 0x41,
				0x47, 0x49, 0x43, 0x2d, 0x43, 0x4f, 0x4f, 0x4b,
				0x49, 0x45, 0x2d, 0x31, 0x00, 0x01, 0x11,
			},
			wantAuthFile: []byte{
				// The order changed, due to new length of auth cookie resulting
				// in remove + append, we verify that the implementation is
				// behaving as expected (changing the order is not a requirement,
				// simply an implementation detail).
				0x01, 0x00, 0x00, 0x01, 0x77, 0x00, 0x01, 0x31,
				0x00, 0x12, 0x4d, 0x49, 0x54, 0x2d, 0x4d, 0x41,
				0x47, 0x49, 0x43, 0x2d, 0x43, 0x4f, 0x4f, 0x4b,
				0x49, 0x45, 0x2d, 0x31, 0x00, 0x01, 0x11,
				0x01, 0x00, 0x00, 0x01, 0x77, 0x00, 0x01, 0x30,
				0x00, 0x12, 0x4d, 0x49, 0x54, 0x2d, 0x4d, 0x41,
				0x47, 0x49, 0x43, 0x2d, 0x43, 0x4f, 0x4f, 0x4b,
				0x49, 0x45, 0x2d, 0x31, 0x00, 0x02, 0xff, 0xff,
			},
			entries: []testEntry{
				{
					address:      "w",
					display:      "0",
					authProtocol: "MIT-MAGIC-COOKIE-1",
					authCookie:   "ffff",
				},
			},
		},
		{
			name: "update entry",
			authFile: []byte{
				// 00000000: 0100 0001 7700 0130 0012 4d49 542d 4d41  ....w..0..MIT-MA
				// 00000010: 4749 432d 434f 4f4b 4945 2d31 0001 0001  GIC-COOKIE-1....
				// 00000020: 0000 0177 0001 3100 124d 4954 2d4d 4147  ...w..1..MIT-MAG
				// 00000030: 4943 2d43 4f4f 4b49 452d 3100 0111       IC-COOKIE-1...
				0x01, 0x00, 0x00, 0x01, 0x77, 0x00, 0x01, 0x30,
				0x00, 0x12, 0x4d, 0x49, 0x54, 0x2d, 0x4d, 0x41,
				0x47, 0x49, 0x43, 0x2d, 0x43, 0x4f, 0x4f, 0x4b,
				0x49, 0x45, 0x2d, 0x31, 0x00, 0x01, 0x00,
				0x01, 0x00, 0x00, 0x01, 0x77, 0x00, 0x01, 0x31,
				0x00, 0x12, 0x4d, 0x49, 0x54, 0x2d, 0x4d, 0x41,
				0x47, 0x49, 0x43, 0x2d, 0x43, 0x4f, 0x4f, 0x4b,
				0x49, 0x45, 0x2d, 0x31, 0x00, 0x01, 0x11,
			},
			wantAuthFile: []byte{
				// 00000000: 0100 0001 7700 0130 0012 4d49 542d 4d41  ....w..0..MIT-MA
				// 00000010: 4749 432d 434f 4f4b 4945 2d31 0001 0001  GIC-COOKIE-1....
				// 00000020: 0000 0177 0001 3100 124d 4954 2d4d 4147  ...w..1..MIT-MAG
				// 00000030: 4943 2d43 4f4f 4b49 452d 3100 0111       IC-COOKIE-1...
				0x01, 0x00, 0x00, 0x01, 0x77, 0x00, 0x01, 0x30,
				0x00, 0x12, 0x4d, 0x49, 0x54, 0x2d, 0x4d, 0x41,
				0x47, 0x49, 0x43, 0x2d, 0x43, 0x4f, 0x4f, 0x4b,
				0x49, 0x45, 0x2d, 0x31, 0x00, 0x01, 0xff,
				0x01, 0x00, 0x00, 0x01, 0x77, 0x00, 0x01, 0x31,
				0x00, 0x12, 0x4d, 0x49, 0x54, 0x2d, 0x4d, 0x41,
				0x47, 0x49, 0x43, 0x2d, 0x43, 0x4f, 0x4f, 0x4b,
				0x49, 0x45, 0x2d, 0x31, 0x00, 0x01, 0x11,
			},
			entries: []testEntry{
				{
					address:      "w",
					display:      "0",
					authProtocol: "MIT-MAGIC-COOKIE-1",
					authCookie:   "ff",
				},
			},
		},
		{
			name: "clean up old entries",
			authFile: []byte{
				// w/unix:0  MIT-MAGIC-COOKIE-1  80507df050756cdefa504b65adb3bcfb
				// w/unix:0  MIT-MAGIC-COOKIE-1  267b37f6cbc11b97beb826bb1aab8570
				// w/unix:0  MIT-MAGIC-COOKIE-1  516e22e2b11d1bd0115dff09c028ca5c
				//
				// 00000000: 0100 0001 7700 0130 0012 4d49 542d 4d41  ....w..0..MIT-MA
				// 00000010: 4749 432d 434f 4f4b 4945 2d31 0010 8050  GIC-COOKIE-1...P
				// 00000020: 7df0 5075 6cde fa50 4b65 adb3 bcfb 0100  }.Pul..PKe......
				// 00000030: 0001 7700 0130 0012 4d49 542d 4d41 4749  ..w..0..MIT-MAGI
				// 00000040: 432d 434f 4f4b 4945 2d31 0010 267b 37f6  C-COOKIE-1..&{7.
				// 00000050: cbc1 1b97 beb8 26bb 1aab 8570 0100 0001  ......&....p....
				// 00000060: 7700 0130 0012 4d49 542d 4d41 4749 432d  w..0..MIT-MAGIC-
				// 00000070: 434f 4f4b 4945 2d31 0010 516e 22e2 b11d  COOKIE-1..Qn"...
				// 00000080: 1bd0 115d ff09 c028 ca5c                 ...]...(.\
				0x01, 0x00, 0x00, 0x01, 0x77, 0x00, 0x01, 0x30,
				0x00, 0x12, 0x4d, 0x49, 0x54, 0x2d, 0x4d, 0x41,
				0x47, 0x49, 0x43, 0x2d, 0x43, 0x4f, 0x4f, 0x4b,
				0x49, 0x45, 0x2d, 0x31, 0x00, 0x10, 0x80, 0x50,
				0x7d, 0xf0, 0x50, 0x75, 0x6c, 0xde, 0xfa, 0x50,
				0x4b, 0x65, 0xad, 0xb3, 0xbc, 0xfb, 0x01, 0x00,
				0x00, 0x01, 0x77, 0x00, 0x01, 0x30, 0x00, 0x12,
				0x4d, 0x49, 0x54, 0x2d, 0x4d, 0x41, 0x47, 0x49,
				0x43, 0x2d, 0x43, 0x4f, 0x4f, 0x4b, 0x49, 0x45,
				0x2d, 0x31, 0x00, 0x10, 0x26, 0x7b, 0x37, 0xf6,
				0xcb, 0xc1, 0x1b, 0x97, 0xbe, 0xb8, 0x26, 0xbb,
				0x1a, 0xab, 0x85, 0x70, 0x01, 0x00, 0x00, 0x01,
				0x77, 0x00, 0x01, 0x30, 0x00, 0x12, 0x4d, 0x49,
				0x54, 0x2d, 0x4d, 0x41, 0x47, 0x49, 0x43, 0x2d,
				0x43, 0x4f, 0x4f, 0x4b, 0x49, 0x45, 0x2d, 0x31,
				0x00, 0x10, 0x51, 0x6e, 0x22, 0xe2, 0xb1, 0x1d,
				0x1b, 0xd0, 0x11, 0x5d, 0xff, 0x09, 0xc0, 0x28,
				0xca, 0x5c,
			},
			wantAuthFile: []byte{
				// w/unix:0  MIT-MAGIC-COOKIE-1  516e5bc892b7162b844abd1fc1a7c16e
				//
				// 00000000: 0100 0001 7700 0130 0012 4d49 542d 4d41  ....w..0..MIT-MA
				// 00000010: 4749 432d 434f 4f4b 4945 2d31 0010 516e  GIC-COOKIE-1..Qn
				// 00000020: 5bc8 92b7 162b 844a bd1f c1a7 c16e       [....+.J.....n
				0x01, 0x00, 0x00, 0x01, 0x77, 0x00, 0x01, 0x30,
				0x00, 0x12, 0x4d, 0x49, 0x54, 0x2d, 0x4d, 0x41,
				0x47, 0x49, 0x43, 0x2d, 0x43, 0x4f, 0x4f, 0x4b,
				0x49, 0x45, 0x2d, 0x31, 0x00, 0x10, 0x51, 0x6e,
				0x5b, 0xc8, 0x92, 0xb7, 0x16, 0x2b, 0x84, 0x4a,
				0xbd, 0x1f, 0xc1, 0xa7, 0xc1, 0x6e,
			},
			entries: []testEntry{
				{
					address:      "w",
					display:      "0",
					authProtocol: "MIT-MAGIC-COOKIE-1",
					authCookie:   "516e5bc892b7162b844abd1fc1a7c16e",
				},
			},
		},
	}

	homedir, err := os.UserHomeDir()
	require.NoError(t, err)

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			fs := afero.NewMemMapFs()
			if tt.authFile != nil {
				err := afero.WriteFile(fs, filepath.Join(homedir, ".Xauthority"), tt.authFile, 0o600)
				require.NoError(t, err)
			}

			for _, entry := range tt.entries {
				err := addXauthEntry(context.Background(), fs, entry.address, entry.display, entry.authProtocol, entry.authCookie)
				require.NoError(t, err)
			}

			gotAuthFile, err := afero.ReadFile(fs, filepath.Join(homedir, ".Xauthority"))
			require.NoError(t, err)

			if diff := cmp.Diff(tt.wantAuthFile, gotAuthFile); diff != "" {
				assert.Failf(t, "addXauthEntry() mismatch", "(-want +got):\n%s", diff)
			}
		})
	}
}
