import { useTheme } from "@emotion/react";
import type {
	HealthcheckReport,
	HealthSeverity,
	NetcheckReport,
} from "api/typesGenerated";
import { Alert } from "components/Alert/Alert";
import { Button } from "components/Button/Button";
import { MapPinIcon } from "lucide-react";
import type { FC } from "react";
import { Link, useOutletContext } from "react-router";
import { pageTitle } from "utils/page";
import {
	BooleanPill,
	Header,
	HeaderTitle,
	HealthMessageDocsLink,
	HealthyDot,
	Logs,
	Main,
	SectionLabel,
} from "./Content";
import { DismissWarningButton } from "./DismissWarningButton";
import { healthyColor } from "./healthyColor";

const flags: BooleanKeys<NetcheckReport>[] = [
	"UDP",
	"IPv6",
	"IPv4",
	"IPv6CanSend",
	"IPv4CanSend",
	"OSHasIPv6",
	"ICMPv4",
	"MappingVariesByDestIP",
	"HairPinning",
	"UPnP",
	"PMP",
	"PCP",
];

type BooleanKeys<T> = {
	[K in keyof T]: T[K] extends boolean | null ? K : never;
}[keyof T];

const DERPPage: FC = () => {
	const { derp } = useOutletContext<HealthcheckReport>();
	const { netcheck, regions, netcheck_logs: logs } = derp;
	const safeNetcheck = netcheck || ({} as NetcheckReport);
	const theme = useTheme();

	return (
		<>
			<title>{pageTitle("DERP - Health")}</title>

			<Header>
				<HeaderTitle>
					<HealthyDot severity={derp.severity as HealthSeverity} />
					DERP
				</HeaderTitle>
				<DismissWarningButton healthcheck="DERP" />
			</Header>

			<Main>
				{derp.warnings.map((warning) => {
					return (
						<Alert
							actions={<HealthMessageDocsLink {...warning} />}
							key={warning.code}
							severity="warning"
							prominent
						>
							{warning.message}
						</Alert>
					);
				})}

				<section>
					<SectionLabel>Flags</SectionLabel>
					<div css={{ display: "flex", flexWrap: "wrap", gap: 12 }}>
						{flags.map((flag) => (
							<BooleanPill key={flag} value={safeNetcheck[flag]}>
								{flag}
							</BooleanPill>
						))}
					</div>
				</section>

				<section>
					<SectionLabel>Regions</SectionLabel>
					<div css={{ display: "flex", flexWrap: "wrap", gap: 12 }}>
						{Object.values(regions ?? {})
							.filter((region) => {
								// Values can technically be null
								return region !== null;
							})
							.sort((a, b) => {
								if (a.region && b.region) {
									return a.region.RegionName.localeCompare(b.region.RegionName);
								}
								return 0;
							})
							.map(({ severity, region }) => {
								return (
									<Button variant="outline" key={region!.RegionID} asChild>
										<Link to={`/health/derp/regions/${region!.RegionID}`}>
											<MapPinIcon
												style={{
													color: healthyColor(
														theme,
														severity as HealthSeverity,
													),
												}}
											/>
											{region!.RegionName}
										</Link>
									</Button>
								);
							})}
					</div>
				</section>

				<section>
					<SectionLabel>Logs</SectionLabel>
					<Logs
						lines={logs}
						css={(theme) => ({
							borderRadius: 8,
							border: `1px solid ${theme.palette.divider}`,
							color: theme.palette.text.secondary,
						})}
					/>
				</section>
			</Main>
		</>
	);
};

export default DERPPage;
