import { css, type Interpolation, type Theme } from "@emotion/react";
import type { APIKeyWithOwner } from "api/typesGenerated";
import { Button } from "components/Button/Button";
import { Stack } from "components/Stack/Stack";
import { PlusIcon } from "lucide-react";
import { type FC, useState } from "react";
import { Link as RouterLink } from "react-router";
import { Section } from "../Section";
import { ConfirmDeleteDialog } from "./ConfirmDeleteDialog";
import { useTokensData } from "./hooks";
import { TokensPageView } from "./TokensPageView";

const cliCreateCommand = "coder tokens create";

const TokensPage: FC = () => {
	const [tokenToDelete, setTokenToDelete] = useState<
		APIKeyWithOwner | undefined
	>(undefined);

	const {
		data: tokens,
		error: getTokensError,
		isFetching,
		isFetched,
		queryKey,
	} = useTokensData({
		// we currently do not show all tokens in the UI, even if
		// the user has read all permissions
		include_all: false,
	});

	return (
		<>
			<Section
				title="Tokens"
				css={styles.section}
				description={
					<>
						Tokens are used to authenticate with the Coder API. You can create a
						token with the Coder CLI using the <code>{cliCreateCommand}</code>{" "}
						command.
					</>
				}
				layout="fluid"
			>
				<TokenActions />
				<TokensPageView
					tokens={tokens}
					isLoading={isFetching}
					hasLoaded={isFetched}
					getTokensError={getTokensError}
					onDelete={(token) => {
						setTokenToDelete(token);
					}}
				/>
			</Section>
			<ConfirmDeleteDialog
				queryKey={queryKey}
				token={tokenToDelete}
				setToken={setTokenToDelete}
			/>
		</>
	);
};

const TokenActions: FC = () => (
	<Stack direction="row" justifyContent="end" css={{ marginBottom: 8 }}>
		<Button asChild variant="outline">
			<RouterLink to="new">
				<PlusIcon />
				Add token
			</RouterLink>
		</Button>
	</Stack>
);

const styles = {
	section: (theme) => css`
    & code {
      background: ${theme.palette.divider};
      font-size: 12px;
      padding: 2px 4px;
      color: ${theme.palette.text.primary};
      border-radius: 2px;
    }
  `,
} satisfies Record<string, Interpolation<Theme>>;

export default TokensPage;
