package com.acme.storage.performance;

import com.google.cloud.storage.Blob;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageOptions;
import com.google.common.base.Stopwatch;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.concurrent.TimeUnit;

import static org.junit.Assert.assertTrue;


public class StoragePerformanceTest {

    private static final Logger log = LoggerFactory.getLogger(StoragePerformanceTest.class);

    private static final BlobId BLOB_ID = BlobId.of(
            "pids-storage-api-qa-blob",
            "5be59fc42ee6c00f00315da5/1096552b-b4ee-4bba-884b-e33758aeee70/random.bin"
    );

    @Test
    public void download() throws Exception {
        Storage storage = StorageOptions.getDefaultInstance().getService();
        Blob blob = storage.get(BLOB_ID);
        Path tempFile = createTempFile();

        Stopwatch stopwatch = Stopwatch.createStarted();
        blob.downloadTo(tempFile);
        long elapsedSeconds = stopwatch.stop().elapsed(TimeUnit.SECONDS);
        double fileSizeMb = blob.getSize().doubleValue() / 1024 / 1024;
        double throughput = fileSizeMb / elapsedSeconds;
        log.info("Completed download: elapsed={}s,fileSize={}MB,throughput={}MB/s",
                elapsedSeconds, fileSizeMb, throughput);
        assertTrue("Expected at least 20MB/s", throughput > 20);
    }

    private Path createTempFile() throws IOException {
        Path tempFile = Files.createTempFile("StoragePerformanceTest", ".bin");
        tempFile.toFile().deleteOnExit();
        log.info("Created temp file: {}", tempFile);
        return tempFile;
    }

}
