<?php

declare(strict_types=1);

namespace Rector\TypeDeclaration\NodeTypeAnalyzer;

use PhpParser\Node\ComplexType;
use PhpParser\Node\Expr;
use PhpParser\Node\Identifier;
use PhpParser\Node\Name;
use PhpParser\Node\Stmt\Property;
use PHPStan\Type\UnionType;
use PHPStan\Type\VerbosityLevel;
use Rector\BetterPhpDocParser\PhpDocInfo\PhpDocInfo;
use Rector\BetterPhpDocParser\PhpDocManipulator\PhpDocTypeChanger;
use Rector\Php\PhpVersionProvider;
use Rector\PhpParser\Node\NodeFactory;
use Rector\PHPStanStaticTypeMapper\TypeAnalyzer\UnionTypeAnalyzer;
use Rector\ValueObject\PhpVersionFeature;

final readonly class PropertyTypeDecorator
{
    public function __construct(
        private UnionTypeAnalyzer $unionTypeAnalyzer,
        private PhpDocTypeChanger $phpDocTypeChanger,
        private PhpVersionProvider $phpVersionProvider,
        private NodeFactory $nodeFactory,
    ) {
    }

    public function decoratePropertyUnionType(
        UnionType $unionType,
        Name|ComplexType|Identifier $typeNode,
        Property $property,
        PhpDocInfo $phpDocInfo,
        bool $changeVarTypeFallback = true
    ): void {
        if (! $this->unionTypeAnalyzer->isNullable($unionType)) {
            if ($this->phpVersionProvider->isAtLeastPhpVersion(PhpVersionFeature::UNION_TYPES)) {
                $property->type = $typeNode;
                return;
            }

            if ($changeVarTypeFallback) {
                $this->phpDocTypeChanger->changeVarType($property, $phpDocInfo, $unionType);
            }

            return;
        }

        $property->type = $typeNode;

        $propertyProperty = $property->props[0];

        // add null default
        if (! $propertyProperty->default instanceof Expr) {
            $propertyProperty->default = $this->nodeFactory->createNull();
        }

        // has array with defined type? add docs
        if (! $this->isDocBlockRequired($unionType)) {
            return;
        }

        if (! $changeVarTypeFallback) {
            return;
        }

        $this->phpDocTypeChanger->changeVarType($property, $phpDocInfo, $unionType);
    }

    private function isDocBlockRequired(UnionType $unionType): bool
    {
        foreach ($unionType->getTypes() as $unionedType) {
            if ($unionedType->isArray()->yes()) {
                $describedArray = $unionedType->describe(VerbosityLevel::value());
                if ($describedArray !== 'array') {
                    return true;
                }
            }
        }

        return false;
    }
}
