"""
The ``temperature`` module contains functions for modeling temperature of
PV modules and cells.
"""

import numpy as np
import pandas as pd
from pvlib.tools import sind
from pvlib._deprecation import warn_deprecated
from pvlib.tools import _get_sample_intervals
import scipy
import warnings


TEMPERATURE_MODEL_PARAMETERS = {
    'sapm': {
        'open_rack_glass_glass': {'a': -3.47, 'b': -.0594, 'deltaT': 3},
        'close_mount_glass_glass': {'a': -2.98, 'b': -.0471, 'deltaT': 1},
        'open_rack_glass_polymer': {'a': -3.56, 'b': -.0750, 'deltaT': 3},
        'insulated_back_glass_polymer': {'a': -2.81, 'b': -.0455, 'deltaT': 0},
    },
    'pvsyst': {'freestanding': {'u_c': 29.0, 'u_v': 0},
               'insulated': {'u_c': 15.0, 'u_v': 0}}
}
"""Dictionary of temperature parameters organized by model.

There are keys for each model at the top level. Currently there are two models,
``'sapm'`` for the Sandia Array Performance Model, and ``'pvsyst'``. Each model
has a dictionary of configurations; a value is itself a dictionary containing
model parameters. Retrieve parameters by indexing the model and configuration
by name. Note: the keys are lower-cased and case sensitive.

Example
-------
Retrieve the open rack glass-polymer configuration for SAPM::

    from pvlib.temperature import TEMPERATURE_MODEL_PARAMETERS
    temperature_model_parameters = (
        TEMPERATURE_MODEL_PARAMETERS['sapm']['open_rack_glass_polymer'])
    # {'a': -3.56, 'b': -0.075, 'deltaT': 3}
"""


def _temperature_model_params(model, parameter_set):
    try:
        params = TEMPERATURE_MODEL_PARAMETERS[model]
        return params[parameter_set]
    except KeyError:
        msg = ('{} is not a named set of parameters for the {} cell'
               ' temperature model.'
               ' See pvlib.temperature.TEMPERATURE_MODEL_PARAMETERS'
               ' for names'.format(parameter_set, model))
        raise KeyError(msg)


def sapm_cell(poa_global, temp_air, wind_speed, a, b, deltaT,
              irrad_ref=1000.):
    r'''
    Calculate cell temperature per the Sandia Array Performance Model.

    See [1]_ for details on the Sandia Array Performance Model.

    Parameters
    ----------
    poa_global : numeric
        Total incident irradiance [W/m^2].

    temp_air : numeric
        Ambient dry bulb temperature [C].

    wind_speed : numeric
        Wind speed at a height of 10 meters [m/s].

    a : float
        Parameter :math:`a` in :eq:`sapm1`.

    b : float
        Parameter :math:`b` in :eq:`sapm1`.

    deltaT : float
        Parameter :math:`\Delta T` in :eq:`sapm2` [C].

    irrad_ref : float, default 1000
        Reference irradiance, parameter :math:`E_{0}` in
        :eq:`sapm2` [W/m^2].

    Returns
    -------
    numeric, values in degrees C.

    Notes
    -----
    The model for cell temperature :math:`T_{C}` is given by a pair of
    equations (Eq. 11 and 12 in [1]_).

    .. math::
       :label: sapm1

       T_{m} = E \times \exp (a + b \times WS) + T_{a}

    .. math::
       :label: sapm2

       T_{C} = T_{m} + \frac{E}{E_{0}} \Delta T

    The module back surface temperature :math:`T_{m}` is implemented in
    :py:func:`~pvlib.temperature.sapm_module`.

    Inputs to the model are plane-of-array irradiance :math:`E` (W/m2) and
    ambient air temperature :math:`T_{a}` (C). Model parameters depend both on
    the module construction and its mounting. Parameter sets are provided in
    [1]_ for representative modules and mounting, and are coded for convenience
    in :data:`~pvlib.temperature.TEMPERATURE_MODEL_PARAMETERS`.

    +---------------+----------------+-------+---------+---------------------+
    | Module        | Mounting       | a     | b       | :math:`\Delta T [C]`|
    +===============+================+=======+=========+=====================+
    | glass/glass   | open rack      | -3.47 | -0.0594 | 3                   |
    +---------------+----------------+-------+---------+---------------------+
    | glass/glass   | close roof     | -2.98 | -0.0471 | 1                   |
    +---------------+----------------+-------+---------+---------------------+
    | glass/polymer | open rack      | -3.56 | -0.075  | 3                   |
    +---------------+----------------+-------+---------+---------------------+
    | glass/polymer | insulated back | -2.81 | -0.0455 | 0                   |
    +---------------+----------------+-------+---------+---------------------+

    References
    ----------
    .. [1] King, D. et al, 2004, "Sandia Photovoltaic Array Performance
       Model", SAND Report 3535, Sandia National Laboratories, Albuquerque,
       NM.

    See also
    --------
    sapm_cell_from_module
    sapm_module

    Examples
    --------
    >>> from pvlib.temperature import sapm_cell, TEMPERATURE_MODEL_PARAMETERS
    >>> params = TEMPERATURE_MODEL_PARAMETERS['sapm']['open_rack_glass_glass']
    >>> sapm_cell(1000, 10, 0, **params)
    44.11703066106086
    '''
    module_temperature = sapm_module(poa_global, temp_air, wind_speed,
                                     a, b)
    return sapm_cell_from_module(module_temperature, poa_global, deltaT,
                                 irrad_ref)


def sapm_module(poa_global, temp_air, wind_speed, a, b):
    r'''
    Calculate module back surface temperature per the Sandia Array
    Performance Model.

    See [1]_ for details on the Sandia Array Performance Model.

    Parameters
    ----------
    poa_global : numeric
        Total incident irradiance [W/m^2].

    temp_air : numeric
        Ambient dry bulb temperature [C].

    wind_speed : numeric
        Wind speed at a height of 10 meters [m/s].

    a : float
        Parameter :math:`a` in :eq:`sapm1mod`.

    b : float
        Parameter :math:`b` in :eq:`sapm1mod`.

    Returns
    -------
    numeric, values in degrees C.

    Notes
    -----
    The model for module temperature :math:`T_{m}` is given by Eq. 11 in [1]_.

    .. math::
       :label: sapm1mod

       T_{m} = E \times \exp (a + b \times WS) + T_{a}

    Inputs to the model are plane-of-array irradiance :math:`E` (W/m2) and
    ambient air temperature :math:`T_{a}` (C). Model outputs are surface
    temperature at the back of the module :math:`T_{m}` and cell temperature
    :math:`T_{C}`. Model parameters depend both on the module construction and
    its mounting. Parameter sets are provided in [1]_ for representative
    modules and mounting, and are coded for convenience in
    :data:`~pvlib.temperature.TEMPERATURE_MODEL_PARAMETERS`.

    +---------------+----------------+-------+---------+---------------------+
    | Module        | Mounting       | a     | b       | :math:`\Delta T [C]`|
    +===============+================+=======+=========+=====================+
    | glass/glass   | open rack      | -3.47 | -0.0594 | 3                   |
    +---------------+----------------+-------+---------+---------------------+
    | glass/glass   | close roof     | -2.98 | -0.0471 | 1                   |
    +---------------+----------------+-------+---------+---------------------+
    | glass/polymer | open rack      | -3.56 | -0.075  | 3                   |
    +---------------+----------------+-------+---------+---------------------+
    | glass/polymer | insulated back | -2.81 | -0.0455 | 0                   |
    +---------------+----------------+-------+---------+---------------------+

    References
    ----------
    .. [1] King, D. et al, 2004, "Sandia Photovoltaic Array Performance
       Model", SAND Report 3535, Sandia National Laboratories, Albuquerque,
       NM.

    See also
    --------
    sapm_cell
    sapm_cell_from_module
    '''
    return poa_global * np.exp(a + b * wind_speed) + temp_air


def sapm_cell_from_module(module_temperature, poa_global, deltaT,
                          irrad_ref=1000.):
    r'''
    Calculate cell temperature from module temperature using the Sandia Array
    Performance Model.

    See [1]_ for details on the Sandia Array Performance Model.

    Parameters
    ----------
    module_temperature : numeric
        Temperature of back of module surface [C].

    poa_global : numeric
        Total incident irradiance [W/m^2].

    deltaT : float
        Parameter :math:`\Delta T` in :eq:`sapm2_cell_from_mod` [C].

    irrad_ref : float, default 1000
        Reference irradiance, parameter :math:`E_{0}` in
        :eq:`sapm2` [W/m^2].

    Returns
    -------
    numeric, values in degrees C.

    Notes
    -----
    The model for cell temperature :math:`T_{C}` is given by Eq. 12 in [1]_.

    .. math::
       :label: sapm2_cell_from_mod

       T_{C} = T_{m} + \frac{E}{E_{0}} \Delta T

    The module back surface temperature :math:`T_{m}` is implemented in
    :py:func:`~pvlib.temperature.sapm_module`.

    Model parameters depend both on the module construction and its mounting.
    Parameter sets are provided in [1]_ for representative modules and
    mounting, and are coded for convenience in
    :data:`~pvlib.temperature.TEMPERATURE_MODEL_PARAMETERS`.

    +---------------+----------------+-------+---------+---------------------+
    | Module        | Mounting       | a     | b       | :math:`\Delta T [C]`|
    +===============+================+=======+=========+=====================+
    | glass/glass   | open rack      | -3.47 | -0.0594 | 3                   |
    +---------------+----------------+-------+---------+---------------------+
    | glass/glass   | close roof     | -2.98 | -0.0471 | 1                   |
    +---------------+----------------+-------+---------+---------------------+
    | glass/polymer | open rack      | -3.56 | -0.075  | 3                   |
    +---------------+----------------+-------+---------+---------------------+
    | glass/polymer | insulated back | -2.81 | -0.0455 | 0                   |
    +---------------+----------------+-------+---------+---------------------+

    References
    ----------
    .. [1] King, D. et al, 2004, "Sandia Photovoltaic Array Performance
       Model", SAND Report 3535, Sandia National Laboratories, Albuquerque,
       NM.

    See also
    --------
    sapm_cell
    sapm_module
    '''
    return module_temperature + (poa_global / irrad_ref) * deltaT


def pvsyst_cell(poa_global, temp_air, wind_speed=1.0, u_c=29.0, u_v=0.0,
                eta_m=None, module_efficiency=0.1, alpha_absorption=0.9):
    r"""
    Calculate cell temperature using an empirical heat loss factor model
    as implemented in PVsyst.

    Parameters
    ----------
    poa_global : numeric
        Total incident irradiance [W/m^2].

    temp_air : numeric
        Ambient dry bulb temperature [C].

    wind_speed : numeric, default 1.0
        Wind speed in m/s measured at the same height for which the wind loss
        factor was determined.  The default value 1.0 m/s is the wind
        speed at module height used to determine NOCT. [m/s]

    u_c : float, default 29.0
        Combined heat loss factor coefficient. The default value is
        representative of freestanding modules with the rear surfaces exposed
        to open air (e.g., rack mounted). Parameter :math:`U_{c}` in
        :eq:`pvsyst`.
        :math:`\left[\frac{\text{W}/{\text{m}^2}}{\text{C}}\right]`

    u_v : float, default 0.0
        Combined heat loss factor influenced by wind. Parameter :math:`U_{v}`
        in :eq:`pvsyst`.
        :math:`\left[ \frac{\text{W}/\text{m}^2}{\text{C}\ \left( \text{m/s} \right)} \right]`  # noQA: E501

    eta_m : numeric, default None (deprecated, use module_efficiency instead)

    module_efficiency : numeric, default 0.1
        Module external efficiency as a fraction. Parameter :math:`\eta_{m}`
        in :eq:`pvsyst`. Calculate as
        :math:`\eta_{m} = DC\ power / (POA\ irradiance \times module\ area)`.

    alpha_absorption : numeric, default 0.9
        Absorption coefficient. Parameter :math:`\alpha` in :eq:`pvsyst`.

    Returns
    -------
    numeric, values in degrees Celsius

    Notes
    -----
    The Pvsyst model for cell temperature :math:`T_{C}` is given by

    .. math::
       :label: pvsyst

        T_{C} = T_{a} + \frac{\alpha E (1 - \eta_{m})}{U_{c} + U_{v} \times WS}

    Inputs to the model are plane-of-array irradiance :math:`E` (W/m2), ambient
    air temperature :math:`T_{a}` (C) and wind speed :math:`WS` (m/s). Model
    output is cell temperature :math:`T_{C}`. Model parameters depend both on
    the module construction and its mounting. Parameters are provided in
    [1]_ for open (freestanding) and close (insulated) mounting configurations,
    , and are coded for convenience in
    :data:`~pvlib.temperature.TEMPERATURE_MODEL_PARAMETERS`. The heat loss
    factors provided represent the combined effect of convection, radiation and
    conduction, and their values are experimentally determined.

    +--------------+---------------+---------------+
    | Mounting     | :math:`U_{c}` | :math:`U_{v}` |
    +==============+===============+===============+
    | freestanding | 29.0          | 0.0           |
    +--------------+---------------+---------------+
    | insulated    | 15.0          | 0.0           |
    +--------------+---------------+---------------+

    References
    ----------
    .. [1] "PVsyst 6 Help", Files.pvsyst.com, 2018. [Online]. Available:
       http://files.pvsyst.com/help/index.html. [Accessed: 10- Dec- 2018].

    .. [2] Faiman, D. (2008). "Assessing the outdoor operating temperature of
       photovoltaic modules." Progress in Photovoltaics 16(4): 307-315.

    Examples
    --------
    >>> from pvlib.temperature import pvsyst_cell, TEMPERATURE_MODEL_PARAMETERS
    >>> params = TEMPERATURE_MODEL_PARAMETERS['pvsyst']['freestanding']
    >>> pvsyst_cell(1000, 10, **params)
    37.93103448275862
    """

    if eta_m:
        warn_deprecated(
            since='v0.9', message='eta_m overwriting module_efficiency',
            name='eta_m', alternative='module_efficiency', removal='v0.10')
        module_efficiency = eta_m
    total_loss_factor = u_c + u_v * wind_speed
    heat_input = poa_global * alpha_absorption * (1 - module_efficiency)
    temp_difference = heat_input / total_loss_factor
    return temp_air + temp_difference


def faiman(poa_global, temp_air, wind_speed=1.0, u0=25.0, u1=6.84):
    r'''
    Calculate cell or module temperature using the Faiman model.

    The Faiman model uses an empirical heat loss factor model [1]_ and is
    adopted in the IEC 61853 standards [2]_ and [3]_.

    Usage of this model in the IEC 61853 standard does not distinguish
    between cell and module temperature.

    Parameters
    ----------
    poa_global : numeric
        Total incident irradiance [W/m^2].

    temp_air : numeric
        Ambient dry bulb temperature [C].

    wind_speed : numeric, default 1.0
        Wind speed in m/s measured at the same height for which the wind loss
        factor was determined.  The default value 1.0 m/s is the wind
        speed at module height used to determine NOCT. [m/s]

    u0 : numeric, default 25.0
        Combined heat loss factor coefficient. The default value is one
        determined by Faiman for 7 silicon modules.
        :math:`\left[\frac{\text{W}/{\text{m}^2}}{\text{C}}\right]`

    u1 : numeric, default 6.84
        Combined heat loss factor influenced by wind. The default value is one
        determined by Faiman for 7 silicon modules.
        :math:`\left[ \frac{\text{W}/\text{m}^2}{\text{C}\ \left( \text{m/s} \right)} \right]`

    Returns
    -------
    numeric, values in degrees Celsius

    Notes
    -----
    All arguments may be scalars or vectors. If multiple arguments
    are vectors they must be the same length.

    References
    ----------
    .. [1] Faiman, D. (2008). "Assessing the outdoor operating temperature of
       photovoltaic modules." Progress in Photovoltaics 16(4): 307-315.

    .. [2] "IEC 61853-2 Photovoltaic (PV) module performance testing and energy
       rating - Part 2: Spectral responsivity, incidence angle and module
       operating temperature measurements". IEC, Geneva, 2018.

    .. [3] "IEC 61853-3 Photovoltaic (PV) module performance testing and energy
       rating - Part 3: Energy rating of PV modules". IEC, Geneva, 2018.

    '''
    # Contributed by Anton Driesse (@adriesse), PV Performance Labs. Dec., 2019

    # The following lines may seem odd since u0 & u1 are probably scalar,
    # but it serves an indirect and easy way of allowing lists and
    # tuples for the other function arguments.
    u0 = np.asanyarray(u0)
    u1 = np.asanyarray(u1)

    total_loss_factor = u0 + u1 * wind_speed
    heat_input = poa_global
    temp_difference = heat_input / total_loss_factor
    return temp_air + temp_difference


def ross(poa_global, temp_air, noct):
    r'''
    Calculate cell temperature using the Ross model.

    The Ross model [1]_ assumes the difference between cell temperature
    and ambient temperature is proportional to the plane of array irradiance,
    and assumes wind speed of 1 m/s. The model implicitly assumes steady or
    slowly changing irradiance conditions.

    Parameters
    ----------
    poa_global : numeric
        Total incident irradiance. [W/m^2]

    temp_air : numeric
        Ambient dry bulb temperature. [C]

    noct : numeric
        Nominal operating cell temperature [C], determined at conditions of
        800 W/m^2 irradiance, 20 C ambient air temperature and 1 m/s wind.

    Returns
    -------
    cell_temperature : numeric
        Cell temperature. [C]

    Notes
    -----
    The Ross model for cell temperature :math:`T_{C}` is given in [1]_ as

    .. math::

        T_{C} = T_{a} + \frac{NOCT - 20}{80} S

    where :math:`S` is the plane of array irradiance in :math:`mW/{cm}^2`.
    This function expects irradiance in :math:`W/m^2`.

    References
    ----------
    .. [1] Ross, R. G. Jr., (1981). "Design Techniques for Flat-Plate
       Photovoltaic Arrays". 15th IEEE Photovoltaic Specialist Conference,
       Orlando, FL.
    '''
    # factor of 0.1 converts irradiance from W/m2 to mW/cm2
    return temp_air + (noct - 20.) / 80. * poa_global * 0.1


def _fuentes_hconv(tave, windmod, tinoct, temp_delta, xlen, tilt,
                   check_reynold):
    # Calculate the convective coefficient as in Fuentes 1987 -- a mixture of
    # free, laminar, and turbulent convection.
    densair = 0.003484 * 101325.0 / tave  # density
    visair = 0.24237e-6 * tave**0.76 / densair  # kinematic viscosity
    condair = 2.1695e-4 * tave**0.84  # thermal conductivity
    reynold = windmod * xlen / visair
    # the boundary between laminar and turbulent is modeled as an abrupt
    # change at Re = 1.2e5:
    if check_reynold and reynold > 1.2e5:
        # turbulent convection
        hforce = 0.0282 / reynold**0.2 * densair * windmod * 1007 / 0.71**0.4
    else:
        # laminar convection
        hforce = 0.8600 / reynold**0.5 * densair * windmod * 1007 / 0.71**0.67
    # free convection via Grashof number
    # NB: Fuentes hardwires sind(tilt) as 0.5 for tilt=30
    grashof = 9.8 / tave * temp_delta * xlen**3 / visair**2 * sind(tilt)
    # product of Nusselt number and (k/l)
    hfree = 0.21 * (grashof * 0.71)**0.32 * condair / xlen
    # combine free and forced components
    hconv = (hfree**3 + hforce**3)**(1/3)
    return hconv


def _hydraulic_diameter(width, height):
    # calculate the hydraulic diameter of a rectangle
    return 2 * (width * height) / (width + height)


def fuentes(poa_global, temp_air, wind_speed, noct_installed, module_height=5,
            wind_height=9.144, emissivity=0.84, absorption=0.83,
            surface_tilt=30, module_width=0.31579, module_length=1.2):
    """
    Calculate cell or module temperature using the Fuentes model.

    The Fuentes model is a first-principles heat transfer energy balance
    model [1]_ that is used in PVWatts for cell temperature modeling [2]_.

    Parameters
    ----------
    poa_global : pandas Series
        Total incident irradiance [W/m^2]

    temp_air : pandas Series
        Ambient dry bulb temperature [C]

    wind_speed : pandas Series
        Wind speed [m/s]

    noct_installed : float
        The "installed" nominal operating cell temperature as defined in [1]_.
        PVWatts assumes this value to be 45 C for rack-mounted arrays and
        49 C for roof mount systems with restricted air flow around the
        module.  [C]

    module_height : float, default 5.0
        The height above ground of the center of the module. The PVWatts
        default is 5.0 [m]

    wind_height : float, default 9.144
        The height above ground at which ``wind_speed`` is measured. The
        PVWatts defauls is 9.144 [m]

    emissivity : float, default 0.84
        The effectiveness of the module at radiating thermal energy. [unitless]

    absorption : float, default 0.83
        The fraction of incident irradiance that is converted to thermal
        energy in the module. [unitless]

    surface_tilt : float, default 30
        Module tilt from horizontal. If not provided, the default value
        of 30 degrees from [1]_ and [2]_ is used. [degrees]

    module_width : float, default 0.31579
        Module width. The default value of 0.31579 meters in combination with
        the default `module_length` gives a hydraulic diameter of 0.5 as
        assumed in [1]_ and [2]_. [m]

    module_length : float, default 1.2
        Module length. The default value of 1.2 meters in combination with
        the default `module_width` gives a hydraulic diameter of 0.5 as
        assumed in [1]_ and [2]_. [m]

    Returns
    -------
    temperature_cell : pandas Series
        The modeled cell temperature [C]

    Notes
    -----
    This function returns slightly different values from PVWatts at night
    and just after dawn. This is because the SAM SSC assumes that module
    temperature equals ambient temperature when irradiance is zero so it can
    skip the heat balance calculation at night.

    References
    ----------
    .. [1] Fuentes, M. K., 1987, "A Simplifed Thermal Model for Flat-Plate
           Photovoltaic Arrays", SAND85-0330, Sandia National Laboratories,
           Albuquerque NM.
           http://prod.sandia.gov/techlib/access-control.cgi/1985/850330.pdf
    .. [2] Dobos, A. P., 2014, "PVWatts Version 5 Manual", NREL/TP-6A20-62641,
           National Renewable Energy Laboratory, Golden CO.
           doi:10.2172/1158421.
    """
    # ported from the FORTRAN77 code provided in Appendix A of Fuentes 1987;
    # nearly all variable names are kept the same for ease of comparison.

    boltz = 5.669e-8
    emiss = emissivity
    absorp = absorption
    xlen = _hydraulic_diameter(module_width, module_length)
    # cap0 has units of [J / (m^2 K)], equal to mass per unit area times
    # specific heat of the module.
    cap0 = 11000
    tinoct = noct_installed + 273.15

    # convective coefficient of top surface of module at NOCT
    windmod = 1.0
    tave = (tinoct + 293.15) / 2
    hconv = _fuentes_hconv(tave, windmod, tinoct, tinoct - 293.15, xlen,
                           surface_tilt, False)

    # determine the ground temperature ratio and the ratio of the total
    # convection to the top side convection
    hground = emiss * boltz * (tinoct**2 + 293.15**2) * (tinoct + 293.15)
    backrat = (
        absorp * 800.0
        - emiss * boltz * (tinoct**4 - 282.21**4)
        - hconv * (tinoct - 293.15)
    ) / ((hground + hconv) * (tinoct - 293.15))
    tground = (tinoct**4 - backrat * (tinoct**4 - 293.15**4))**0.25
    tground = np.clip(tground, 293.15, tinoct)

    tgrat = (tground - 293.15) / (tinoct - 293.15)
    convrat = (absorp * 800 - emiss * boltz * (
        2 * tinoct**4 - 282.21**4 - tground**4)) / (hconv * (tinoct - 293.15))

    # adjust the capacitance (thermal mass) of the module based on the INOCT.
    # It is a function of INOCT because high INOCT implies thermal coupling
    # with the racking (e.g. roofmount), so the thermal mass is increased.
    # `cap` has units J/(m^2 C) -- see Table 3, Equations 26 & 27
    cap = cap0
    if tinoct > 321.15:
        cap = cap * (1 + (tinoct - 321.15) / 12)

    # iterate through timeseries inputs
    sun0 = 0
    tmod0 = 293.15

    # n.b. the way Fuentes calculates the first timedelta makes it seem like
    # the value doesn't matter -- rather than recreate it here, just assume
    # it's the same as the second timedelta:
    timedelta_seconds = poa_global.index.to_series().diff().dt.total_seconds()
    timedelta_hours = timedelta_seconds / 3600
    timedelta_hours.iloc[0] = timedelta_hours.iloc[1]

    tamb_array = temp_air + 273.15
    sun_array = poa_global * absorp

    # Two of the calculations are easily vectorized, so precalculate them:
    # sky temperature -- Equation 24
    tsky_array = 0.68 * (0.0552 * tamb_array**1.5) + 0.32 * tamb_array
    # wind speed at module height -- Equation 22
    # not sure why the 1e-4 factor is included -- maybe the equations don't
    # behave well if wind == 0?
    windmod_array = wind_speed * (module_height/wind_height)**0.2 + 1e-4

    tmod0 = 293.15
    tmod_array = np.zeros_like(poa_global)

    iterator = zip(tamb_array, sun_array, windmod_array, tsky_array,
                   timedelta_hours)
    for i, (tamb, sun, windmod, tsky, dtime) in enumerate(iterator):
        # solve the heat transfer equation, iterating because the heat loss
        # terms depend on tmod. NB Fuentes doesn't show that 10 iterations is
        # sufficient for convergence.
        tmod = tmod0
        for j in range(10):
            # overall convective coefficient
            tave = (tmod + tamb) / 2
            hconv = convrat * _fuentes_hconv(tave, windmod, tinoct,
                                             abs(tmod-tamb), xlen,
                                             surface_tilt, True)
            # sky radiation coefficient (Equation 3)
            hsky = emiss * boltz * (tmod**2 + tsky**2) * (tmod + tsky)
            # ground radiation coeffieicient (Equation 4)
            tground = tamb + tgrat * (tmod - tamb)
            hground = emiss * boltz * (tmod**2 + tground**2) * (tmod + tground)
            # thermal lag -- Equation 8
            eigen = - (hconv + hsky + hground) / cap * dtime * 3600
            # not sure why this check is done, maybe as a speed optimization?
            if eigen > -10:
                ex = np.exp(eigen)
            else:
                ex = 0
            # Equation 7 -- note that `sun` and `sun0` already account for
            # absorption (alpha)
            tmod = tmod0 * ex + (
                (1 - ex) * (
                    hconv * tamb
                    + hsky * tsky
                    + hground * tground
                    + sun0
                    + (sun - sun0) / eigen
                ) + sun - sun0
            ) / (hconv + hsky + hground)
        tmod_array[i] = tmod
        tmod0 = tmod
        sun0 = sun

    return pd.Series(tmod_array - 273.15, index=poa_global.index, name='tmod')


def _adj_for_mounting_standoff(x):
    # supports noct cell temperature function. Except for x > 3.5, the SAM code
    # and documentation aren't clear on the precise intervals. The choice of
    # < or <= here is pvlib's.
    return np.piecewise(x, [x <= 0, (x > 0) & (x < 0.5),
                            (x >= 0.5) & (x < 1.5), (x >= 1.5) & (x < 2.5),
                            (x >= 2.5) & (x <= 3.5), x > 3.5],
                        [0., 18., 11., 6., 2., 0.])


def noct_sam(poa_global, temp_air, wind_speed, noct, module_efficiency,
             effective_irradiance=None, transmittance_absorptance=0.9,
             array_height=1, mount_standoff=4):
    r'''
    Cell temperature model from the System Advisor Model (SAM).

    The model is described in [1]_, Section 10.6.

    Parameters
    ----------
    poa_global : numeric
        Total incident irradiance. [W/m^2]

    temp_air : numeric
        Ambient dry bulb temperature. [C]

    wind_speed : numeric
        Wind speed in m/s measured at the same height for which the wind loss
        factor was determined.  The default value 1.0 m/s is the wind
        speed at module height used to determine NOCT. [m/s]

    noct : float
        Nominal operating cell temperature [C], determined at conditions of
        800 W/m^2 irradiance, 20 C ambient air temperature and 1 m/s wind.

    module_efficiency : float
        Module external efficiency [unitless] at reference conditions of
        1000 W/m^2 and 20C. Denoted as :math:`eta_{m}` in [1]_. Calculate as
        :math:`\eta_{m} = \frac{V_{mp} I_{mp}}{A \times 1000 W/m^2}`
        where A is module area [m^2].

    effective_irradiance : numeric, default None.
        The irradiance that is converted to photocurrent. If None,
        assumed equal to poa_global. [W/m^2]

    transmittance_absorptance : numeric, default 0.9
        Coefficient for combined transmittance and absorptance effects.
        [unitless]

    array_height : int, default 1
        Height of array above ground in stories (one story is about 3m). Must
        be either 1 or 2. For systems elevated less than one story, use 1.
        If system is elevated more than two stories, use 2.

    mount_standoff : numeric, default 4
        Distance between array mounting and mounting surface. Use default
        if system is ground-mounted. [inches]

    Returns
    -------
    cell_temperature : numeric
        Cell temperature. [C]

    Raises
    ------
    ValueError
        If array_height is an invalid value (must be 1 or 2).

    References
    ----------
    .. [1] Gilman, P., Dobos, A., DiOrio, N., Freeman, J., Janzou, S.,
           Ryberg, D., 2018, "SAM Photovoltaic Model Technical Reference
           Update", National Renewable Energy Laboratory Report
           NREL/TP-6A20-67399.
    '''
    # in [1] the denominator for irr_ratio isn't precisely clear. From
    # reproducing output of the SAM function noct_celltemp_t, we determined
    # that:
    #  - G_total (SAM) is broadband plane-of-array irradiance before
    #    reflections. Equivalent to pvlib variable poa_global
    #  - Geff_total (SAM) is POA irradiance after reflections and
    #    adjustment for spectrum. Equivalent to effective_irradiance
    if effective_irradiance is None:
        irr_ratio = 1.
    else:
        irr_ratio = effective_irradiance / poa_global

    if array_height == 1:
        wind_adj = 0.51 * wind_speed
    elif array_height == 2:
        wind_adj = 0.61 * wind_speed
    else:
        raise ValueError(
            f'array_height must be 1 or 2, {array_height} was given')

    noct_adj = noct + _adj_for_mounting_standoff(mount_standoff)
    tau_alpha = transmittance_absorptance * irr_ratio

    # [1] Eq. 10.37 isn't clear on exactly what "G" is. SAM SSC code uses
    # poa_global where G appears
    cell_temp_init = poa_global / 800. * (noct_adj - 20.)
    heat_loss = 1 - module_efficiency / tau_alpha
    wind_loss = 9.5 / (5.7 + 3.8 * wind_adj)
    return temp_air + cell_temp_init * heat_loss * wind_loss


def prilliman(temp_cell, wind_speed, unit_mass=11.1, coefficients=None):
    """
    Smooth short-term cell temperature transients using the Prilliman model.

    The Prilliman et al. model [1]_ applies a weighted moving average to
    the output of a steady-state cell temperature model to account for
    a module's thermal inertia by smoothing the cell temperature's
    response to changing weather conditions.

    .. warning::
        This implementation requires the time series inputs to be regularly
        sampled in time with frequency less than 20 minutes.  Data with
        irregular time steps should be resampled prior to using this function.

    Parameters
    ----------
    temp_cell : pandas.Series with DatetimeIndex
        Cell temperature modeled with steady-state assumptions. [C]

    wind_speed : pandas.Series
        Wind speed, adjusted to correspond to array height [m/s]

    unit_mass : float, default 11.1
        Total mass of module divided by its one-sided surface area [kg/m^2]

    coefficients : 4-element list-like, optional
        Values for coefficients a_0 through a_3, see Eq. 9 of [1]_

    Returns
    -------
    temp_cell : pandas.Series
        Smoothed version of the input cell temperature. Input temperature
        with sampling interval >= 20 minutes is returned unchanged. [C]

    Notes
    -----
    This smoothing model was developed and validated using the SAPM
    cell temperature model for the steady-state input.

    Smoothing is done using the 20 minute window behind each temperature
    value. At the beginning of the series where a full 20 minute window is not
    possible, partial windows are used instead.

    Output ``temp_cell[k]`` is NaN when input ``wind_speed[k]`` is NaN, or
    when no non-NaN data are in the input temperature for the 20 minute window
    preceding index ``k``.

    References
    ----------
    .. [1] M. Prilliman, J. S. Stein, D. Riley and G. Tamizhmani,
       "Transient Weighted Moving-Average Model of Photovoltaic Module
       Back-Surface Temperature," IEEE Journal of Photovoltaics, 2020.
       :doi:`10.1109/JPHOTOV.2020.2992351`
    """

    # `sample_interval` in minutes:
    sample_interval, samples_per_window = \
        _get_sample_intervals(times=temp_cell.index, win_length=20)

    if sample_interval >= 20:
        warnings.warn("temperature.prilliman only applies smoothing when "
                      "the sampling interval is shorter than 20 minutes "
                      f"(input sampling interval: {sample_interval} minutes);"
                      " returning input temperature series unchanged")
        # too coarsely sampled for smoothing to be relevant
        return temp_cell

    # handle cases where the time series is shorter than 20 minutes total
    samples_per_window = min(samples_per_window, len(temp_cell))

    # prefix with NaNs so that the rolling window is "full",
    # even for the first actual value:
    prefix = np.full(samples_per_window, np.nan)
    temp_cell_prefixed = np.append(prefix, temp_cell.values)

    # generate matrix of integers for creating windows with indexing
    H = scipy.linalg.hankel(np.arange(samples_per_window),
                            np.arange(samples_per_window - 1,
                                      len(temp_cell_prefixed) - 1))
    # each row of `subsets` is the values in one window
    subsets = temp_cell_prefixed[H].T

    # `subsets` now looks like this (for 5-minute data, so 4 samples/window)
    # where "1." is a stand-in for the actual temperature values
    # [[nan, nan, nan, nan],
    #  [nan, nan, nan,  1.],
    #  [nan, nan,  1.,  1.],
    #  [nan,  1.,  1.,  1.],
    #  [ 1.,  1.,  1.,  1.],
    #  [ 1.,  1.,  1.,  1.],
    #  [ 1.,  1.,  1.,  1.],
    #  ...

    # calculate weights for the values in each window
    if coefficients is not None:
        a = coefficients
    else:
        # values from [1], Table II
        a = [0.0046, 0.00046, -0.00023, -1.6e-5]

    wind_speed = wind_speed.values
    p = a[0] + a[1]*wind_speed + a[2]*unit_mass + a[3]*wind_speed*unit_mass
    # calculate the time lag for each sample in the window, paying attention
    # to units (seconds for `timedeltas`, minutes for `sample_interval`)
    timedeltas = np.arange(samples_per_window, 0, -1) * sample_interval * 60
    weights = np.exp(-p[:, np.newaxis] * timedeltas)

    # Set weights corresponding to the prefix values to zero; otherwise the
    # denominator of the weighted average below would be wrong.
    # Weights corresponding to (non-prefix) NaN values must be zero too
    # for the same reason.

    # Right now `weights` is something like this
    # (using 5-minute inputs, so 4 samples per window -> 4 values per row):
    # [[0.0611, 0.1229, 0.2472, 0.4972],
    #  [0.0611, 0.1229, 0.2472, 0.4972],
    #  [0.0611, 0.1229, 0.2472, 0.4972],
    #  [0.0611, 0.1229, 0.2472, 0.4972],
    #  [0.0611, 0.1229, 0.2472, 0.4972],
    #  [0.0611, 0.1229, 0.2472, 0.4972],
    #  [0.0611, 0.1229, 0.2472, 0.4972],
    #  ...

    # After the next line, the NaNs in `subsets` will be zeros in `weights`,
    # like this (with more zeros for any NaNs in the input temperature):

    # [[0.    , 0.    , 0.    , 0.    ],
    #  [0.    , 0.    , 0.    , 0.4972],
    #  [0.    , 0.    , 0.2472, 0.4972],
    #  [0.    , 0.1229, 0.2472, 0.4972],
    #  [0.0611, 0.1229, 0.2472, 0.4972],
    #  [0.0611, 0.1229, 0.2472, 0.4972],
    #  [0.0611, 0.1229, 0.2472, 0.4972],
    #  ...

    weights[np.isnan(subsets)] = 0

    # change the first row of weights from zero to nan -- this is a
    # trick to prevent div by zero warning when dividing by summed weights
    weights[0, :] = np.nan

    # finally, take the weighted average of each window:
    # use np.nansum for numerator to ignore nans in input temperature, but
    # np.sum for denominator to propagate nans in input wind speed.
    numerator = np.nansum(subsets * weights, axis=1)
    denominator = np.sum(weights, axis=1)
    smoothed = numerator / denominator
    smoothed[0] = temp_cell.values[0]
    smoothed = pd.Series(smoothed, index=temp_cell.index)
    return smoothed
