// @flow

import type {
  SourcePacket,
  ResumedPacket,
  PausedPacket,
  ThreadClient,
  Actions
} from "./types";

import { createPause, createSource } from "./create";
import { isEnabled } from "devtools-config";

const CALL_STACK_PAGE_SIZE = 1000;

type Dependencies = {
  threadClient: ThreadClient,
  actions: Actions,
  supportsWasm: boolean
};

let threadClient: ThreadClient;
let actions: Actions;
let supportsWasm: boolean;

function setupEvents(dependencies: Dependencies) {
  threadClient = dependencies.threadClient;
  actions = dependencies.actions;
  supportsWasm = dependencies.supportsWasm;

  if (threadClient) {
    Object.keys(clientEvents).forEach(eventName => {
      threadClient.addListener(eventName, clientEvents[eventName]);
    });
  }
}

async function paused(_: "paused", packet: PausedPacket) {
  // If paused by an explicit interrupt, which are generated by the
  // slow script dialog and internal events such as setting
  // breakpoints, ignore the event.
  const { why } = packet;
  if (why.type === "interrupted" && !packet.why.onNext) {
    return;
  }

  // Eagerly fetch the frames
  const response = await threadClient.getFrames(0, CALL_STACK_PAGE_SIZE);

  if (why.type != "alreadyPaused") {
    const pause = createPause(packet, response);
    actions.paused(pause);
  }
}

function resumed(_: "resumed", packet: ResumedPacket) {
  actions.resumed(packet);
}

function newSource(_: "newSource", { source }: SourcePacket) {
  actions.newSource(createSource(source, { supportsWasm }));

  if (isEnabled("eventListeners")) {
    actions.fetchEventListeners();
  }
}

const clientEvents = {
  paused,
  resumed,
  newSource
};

export { setupEvents, clientEvents };
