"""
This script generates an html table of contributors, with names and avatars.
The list is generated from scikit-learn's teams on GitHub, plus a small number
of hard-coded contributors.

The table should be updated for each new inclusion in the teams.
Generating the table requires admin rights.
"""
from __future__ import print_function

import sys
import requests
import getpass

try:
    # With authentication: up to 5000 requests per hour.
    print("user:", file=sys.stderr)
    user = input()
    passwd = getpass.getpass()
    auth = (user, passwd)
except IndexError:
    # Without authentication: up to 60 requests per hour.
    auth = None

ROW_SIZE = 7
LOGO_URL = 'https://avatars2.githubusercontent.com/u/365630?v=4'


def group_iterable(iterable, size):
    """Group iterable into lines"""
    group = []
    for element in iterable:
        group.append(element)
        if len(group) == size:
            yield group
            group = []
    if len(group) != 0:
        yield group


def get_contributors():
    """Get the list of contributor profiles. Require admin rights."""
    # get members of scikit-learn teams on GitHub
    members = []
    for team in [11523, 33471]:
        for page in [1, 2]:  # 30 per page
            members.extend(requests.get(
                "https://api.github.com/teams/%d/members?page=%d"
                % (team, page), auth=auth).json())

    # keep only the logins
    logins = [c['login'] for c in members]
    # add missing contributors with GitHub accounts
    logins.extend(['dubourg', 'jarrodmillman', 'mbrucher', 'thouis'])
    # add missing contributors without GitHub accounts
    logins.extend(['Angel Soler Gollonet'])
    # remove duplicate
    logins = set(logins)
    # remove CI
    logins.remove('sklearn-ci')

    # get profiles from GitHub
    profiles = [get_profile(login) for login in logins]
    # sort by last name
    profiles = sorted(profiles, key=key)

    return profiles


def get_profile(login):
    """Get the GitHub profile from login"""
    profile = requests.get("https://api.github.com/users/%s" % login,
                           auth=auth).json()
    if 'name' not in profile:
        # default profile if the login does not exist
        return dict(name=login, avatar_url=LOGO_URL, html_url="")
    else:
        if profile["name"] is None:
            profile["name"] = profile["login"]

        # fix missing names
        missing_names = {'bthirion': 'Bertrand Thirion',
                         'dubourg': 'Vincent Dubourg',
                         'Duchesnay': 'Edouard Duchesnay',
                         'Lars': 'Lars Buitinck',
                         'MechCoder': 'Manoj Kumar'}
        if profile["name"] in missing_names:
            profile["name"] = missing_names[profile["name"]]
        return profile


def key(profile):
    """Get the last name in lower case"""
    return profile["name"].split(' ')[-1].lower()


contributors = get_contributors()

print(".. raw :: html\n")
print("    <!-- Generated by generate_authors_table.py -->")
print("    <table>")
print("    <col style='width:%d%%' span='%d'>"
      % (int(100 / ROW_SIZE), ROW_SIZE))
print("    <style>")
print("      img.avatar {border-radius: 10px;}")
print("      td {vertical-align: top;}")
print("    </style>")
for row in group_iterable(contributors, size=ROW_SIZE):
    print("    <tr>")
    for contributor in row:
        print("    <td>")
        print("    <a href='%s'><img src='%s' class='avatar' /></a> <br />"
              % (contributor["html_url"], contributor["avatar_url"]))
        print("    <p>%s</p>" % contributor["name"])
        print("    </td>")
    print("    </tr>")
print("    </table>")
